/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.change;

import java.util.Collections;
import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;
import nz.org.riskscape.wizard.bld.dsl.IncompletePipelineChange;

/**
 * A pipeline change that allows multiple individual changes to be combined in to one.  Not sure if this class has a
 * future, but keeping it for now until the original survey has been ported.
 *
 */
@RequiredArgsConstructor
public class MultipleChanges implements PipelineChange {

  @Getter
  private final Answer answer;

  private final List<IncompletePipelineChange> children;

  @Override
  public ResultOrProblems<IncrementalBuildState> make(IncrementalBuildState currentState)
      throws BadPipelineChangeException {

    // NB potential problems here:
    // - the name of the question is now wrong, which might cause a look up to fail (although the last question
    // should be looked up with getLastQuestion
    // the reason we have to rename the question is to allow the append to work against the empty state, although
    // we could potentially sort that with a nonce question name at the beginning...  yep, make that change, seems safer

    Question nonceQuestion = new Question("remove-me", Void.class).optionalOne().inSet(answer.getQuestionSet());
    Answer appendAnswer = new Answer(nonceQuestion, Collections.emptyList());
    PipelineChange noChange = new NoChange(appendAnswer);
    IncrementalBuildState appendTo =
        currentState.append(noChange, currentState.getAst(), currentState.getRealizedPipeline());

    int index = 0;
    for (IncompletePipelineChange pipelineChange : children) {
      ResultOrProblems<IncrementalBuildState> updatedOr;
      try {
        updatedOr = pipelineChange.apply(answer).make(appendTo);
      } catch (BadPipelineChangeException ex) {
        throw new BadPipelineChangeException(this, "bad change from child " + index, ex);
      }

      if (updatedOr.hasErrors()) {
        // TODO wrap with some context
        return updatedOr;
      } else {
        IncrementalBuildState updated = updatedOr.get();
        appendTo = currentState.append(noChange, updated.getAst(), updated.getRealizedPipeline());
      }
    }

    // throw away the intermediate changes
    return ResultOrProblems.of(currentState.append(this, appendTo.getAst(), appendTo.getRealizedPipeline()));
  }

  @Override
  public String toString() {
    return BasePipelineChange.toString(getClass(), answer, null);
  }
}
