/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.change;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;

/**
 * Base class for any {@link PipelineChange} that follows a standard `string -> ast -> validate -> update` pattern
 */
@RequiredArgsConstructor
public abstract class BasePipelineChange<T> implements PipelineChange {

  /**
   * Template for a pipeline change tostring
   */
  public static String toString(Class<?> clazz, Answer answer, String extra) {
    if (extra != null) {
      extra = " " +  extra;
    }
    return String.format("%s[question=%s responses=%s%s]",
        clazz.getSimpleName(),
        answer.getQuestion().getId(),
        answer.getOriginalResponses(),
        extra
      );
  }

  @Getter
  private final Answer answer;

  @Override
  public final ResultOrProblems<IncrementalBuildState> make(IncrementalBuildState state) {
    PipelineDeclaration astSegment = parse(getSource());

    T validated = validateParsedSegment(astSegment);

    PipelineDeclaration updatedAst = updateAst(state, astSegment, validated);

    return state.realizePipeline(updatedAst).map(rp -> state.append(this, updatedAst, rp));
  }

  /**
   * Validate that the parsed pipeline segment is ok to use in this kind of change.
   * @return the interesting bit of the pipeline, e.g. a single step, a step chain, whatever
   * @throws BadPipelineChangeException if it's invalid - it has been coded badly (not a user error)
   */
  protected abstract T validateParsedSegment(PipelineDeclaration declaration) throws BadPipelineChangeException;


  /**
   * Build a new AST that is the result of this change
   * @param state the previous build state that we are building upon
   * @param astSegment the segment that was parsed and validated.  This is *not* the previous state's ast.
   * @param validated the chunk of the parsed segment that the validate routine method returned - this is often the
   * node of interest that ultimately gets added to the previous state's ast
   * @return a pipeline ast to be used in the next {@link IncrementalBuildState}
   */
  protected abstract PipelineDeclaration updateAst(
      IncrementalBuildState state,
      PipelineDeclaration astSegment,
      T validated
  );

  /**
   * TODO maybe break this up in to something else?
   * @return the source code that was assembled to be parsed in to a {@link PipelineDeclaration}
   */
  protected abstract String getSource();

  @Override
  public String toString() {
    return toString(getClass(), answer, "source=" + getSource());
  }
}
