/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.change;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepChain;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;
import nz.org.riskscape.wizard.bld.loc.ChangeLocation;

public class AppendJoinChange implements PipelineChange {

  public enum Side {
    LHS,
    RHS
  }

  @Getter
  private final Answer answer;
  private final StepChain mainline;
  private final StepChain otherChain;
  private final ChangeLocation otherSide;

  @RequiredArgsConstructor
  public static class Builder {

    private final Answer answer;

    private StepChain mainline;
    private StepChain otherChain;
    private ChangeLocation otherSide;

    public AppendJoinChange build() {
      return new AppendJoinChange(this);
    }

    public Builder append(String setMainline, Object... formatArgs) {
      this.mainline = PipelineParser.INSTANCE.parsePipeline(String.format(setMainline, formatArgs)).getFirst();
      return this;
    }

    public Builder append(StepChain setMainline) {
      this.mainline = setMainline;
      return this;
    }

    public Builder joins(ChangeLocation setOtherSide, String setOtherChain, Object... formatArgs) {
      this.otherChain = PipelineParser.INSTANCE.parsePipeline(String.format(setOtherChain, formatArgs)).getFirst();
      this.otherSide = setOtherSide;
      return this;
    }

    public Builder joins(ChangeLocation setOtherSide, StepChain setOtherChain) {
      this.otherChain = setOtherChain;
      this.otherSide = setOtherSide;
      return this;
    }
  }

  public static Builder builder(Answer answer) {
    return new Builder(answer);
  }

  private AppendJoinChange(Builder builder) {
    this.answer = builder.answer;

    this.mainline = builder.mainline;
    this.otherChain = builder.otherChain;
    this.otherSide = builder.otherSide;
  }

  @Override
  public ResultOrProblems<IncrementalBuildState> make(IncrementalBuildState currentState)
      throws BadPipelineChangeException {
    PipelineDeclaration ast = currentState.getAst();

    StepChain otherSideLocation = otherSide.findOne(currentState).get().getChain();
    StepChain mainlineLocation = answer.getQuestion().getChangeLocation().findOne(currentState).get().getChain();

    if (otherSideLocation.equals(mainlineLocation)) {
      throw new BadPipelineChangeException(this, "can not join the same branch to itself");
    }

    PipelineDeclaration astWithOther = ast.replace(otherSideLocation, otherSideLocation.append(otherChain));
    PipelineDeclaration newAst = astWithOther.replace(mainlineLocation, mainlineLocation.append(mainline));

    return currentState.realizePipeline(newAst).map(rp -> currentState.append(this, newAst, rp));
  }

  @Override
  public String toString() {
    return BasePipelineChange.toString(getClass(), answer, "location=" + otherSide);
  }

}
