/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;

import lombok.Getter;
import nz.org.riskscape.engine.model.ModelParameter;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterProperties;
import nz.org.riskscape.engine.bind.ParameterProperty;
import nz.org.riskscape.engine.bind.ParameterTemplate;
import nz.org.riskscape.engine.bind.ParameterTemplate.Choice;
import nz.org.riskscape.engine.i18n.TranslationContext;

/**
 * A {@link ModelParameter} that is specific to a wizard framework model.
 */
public class WizardModelParameter extends ModelParameter {

  @Getter
  private final Question question;

  private final IncrementalBuildState buildState;

  public WizardModelParameter(Parameter parameter, Question question, IncrementalBuildState buildState) {
    super(parameter, ParameterTemplate.EMPTY);
    this.question = question;
    this.buildState = buildState;
  }

  public WizardModelParameter(Question question, IncrementalBuildState buildState) {
    this(question.toParameter(), question, buildState);
  }

  @Override
  public String getDescription(Locale locale) {
    return getMessage("title", locale)
        .map(str -> str + " " + getMessage("description", locale).orElse(""))
        .orElse("");
  }

  @Override
  protected String toUserFriendlyString(String input) {
    // wizard parameter IDs also have '.'s separating the question-sets
    // Transforms "impact.sort-by[0][direction]" to "Impact sort by 0 direction"
    return super.toUserFriendlyString(input).replaceAll("[\\.\\[\\]]+", " ").trim();
  }

  @Override
  public String getLabel() {
    return toUserFriendlyString(getParameter().getName());
  }

  @Override
  public List<Choice> getChoices(Locale locale) {
    TranslationContext context = new TranslationContext(locale, question.getMessageSource());
    return question.getChoices(buildState).stream().map(c -> c.toParameter(context)).toList();
  }

  @Override
  public Set<ParameterProperty> getProperties() {
    return ParameterProperties.infer(question.getParameterType());
  }

  public Optional<String> getMessage(String suffix, Locale locale) {
    return question.getMessage(suffix, locale);
  }
}
