/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.bind.ParameterTemplate;
import nz.org.riskscape.engine.i18n.TranslationContext;

/**
 * Represents a predefined option, or 'choice', that the user can make when configuring
 * model parameter or wizard question.
 */
@RequiredArgsConstructor @EqualsAndHashCode
public class Choice {

  /**
   * The actual value that gets bound and turned into pipeline code
   */
  @Getter
  private final String value;

  /**
   * A friendly label for the choice
   */
  private final String label;

  /**
   * Extra help information to describe the choice
   */
  protected final Optional<String> description;

  // just in case, we can keep the original thing, e.g. a function or an enum object
  private final Object derivedFrom;

  public String getDescription(TranslationContext context) {
    return description.orElse("");
  }

  public String getLabel(TranslationContext context) {
    return label;
  }

  /**
   * @return a choice that can be passed up to the Platform UI
   */
  public ParameterTemplate.Choice toParameter(TranslationContext context) {
    return new ParameterTemplate.Choice(getValue(), Optional.of(getLabel(context)));
  }

   /**
   * Convenience to return the underlying object in the correct type
   */
  public <T> T getDerivedFrom(Class<T> asType) {
    return isA(asType)
        // developer coding error
        .orElseThrow(() -> new ClassCastException(derivedFrom + " is not a " + asType));
  }

  /**
   * Checks if the underlying object is of the given type, and returns it if so
   */
  public <T> Optional<T> isA(Class<T> asType) {
    if (asType.isAssignableFrom(derivedFrom.getClass())) {
      return Optional.of(asType.cast(derivedFrom));
    }
    return Optional.empty();
  }

}
