/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import java.util.Locale;
import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;

@EqualsAndHashCode(of = {"survey", "id"})
public abstract class BaseQuestionSet implements QuestionSet {

  @Getter
  protected final String name;

  protected final String discriminator;

  @Getter
  protected final String id;

  @Getter
  protected final Survey survey;

  /**
   * Construct a question set where the id is used for the i18n look up.
   */
  public BaseQuestionSet(String id, Survey survey) {
    this.id = id;
    this.survey = survey;
    this.discriminator = null;
    this.name = id;
  }

  /**
   * Construct a question set where the name is used for the i18n look up and the id is set as the name plus an integer
   * disriminator, e.g. `{name: reports, discriminator: 2}` would give an id of `reports-2`
   */
  public BaseQuestionSet(String name, String discriminator, Survey survey) {
    this.id = name + "-" + discriminator;
    this.survey = survey;
    this.discriminator = discriminator;
    this.name = name;
  }

  private String getI18nMessage(String key, Locale locale, Object... args) {
    String i18nMsg = null;

    // first with the discriminator appended, to see if there's a more precise message
    if (discriminator != null) {
      i18nMsg = getSurvey().getMessageSource().getMessage(key + "." + discriminator, args, locale);
    }

    if (i18nMsg == null) {
      // no match, fallback to just the key itself
      i18nMsg = getSurvey().getMessageSource().getMessage(key, args, locale);
    }
    return i18nMsg;
  }

  @Override
  public Optional<String> getDescription(Locale locale) {
    return Optional.ofNullable(
        getI18nMessage("set." + name + ".description", locale, discriminator)
    );
  }

  @Override
  public Optional<String> getTitle(Locale locale) {
    return Optional.ofNullable(
        getI18nMessage("set." + name + ".title", locale, discriminator)
    );
  }
}
