/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import java.util.List;
import java.util.function.Function;

import lombok.Data;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer.Response;

/**
 * An AskAs is a way for a Question to signal to the UI that it can ask the question in a customized way, but ultimately
 * the survey wants the answer to be recorded in a way that can be bound to the Question's parameter type.  This allows
 * a user-friendly experience for certain options while being able to still allow a more complex, custom value to be
 * edited in "expert mode" scenarios (such as editing an ini file, or switching a GUI in to expert mode)
 *
 * NB at the moment, ask as questions are assumed to always be composite, as they are the only examples we've seen so
 * far.  There's no reason for it to need to be this way, but it was simpler at the time.
 */
@Data
public class AskAs<T> {
  private final Class<T> type;
  private final Function<T, String> toAnswer;

  public String toAnswer(T delegatedAskerBoundValue) {
    return toAnswer.apply(delegatedAskerBoundValue);
  }

  /**
   * Convert the result of asking the question ({@link #getType()}) to the type the survey wants to have stored
   * ({@link Question#getType()}
   */
  public ResultOrProblems<Response> convert(BindingContext context, Question question, Response response) {
    T askedBoundValue = type.cast(response.getBoundValue());
    String convertedInput = toAnswer(askedBoundValue);
    return Answer.bind(context, question, convertedInput);
  }

  /**
   * @return a modified copy of parent with the parametertype set to the ask as type and with the ask as information
   * removed
   */
  public Question asType(Question parent) {
    return parent.withType(getType()).withAskAs(null);
  }

  /**
   * @return a list of questions that can be used to build a response to this question when 'asked as'
   */
  public List<Question> getCompositeSubparts(Question parent) {
    return Question.getCompositeSubparts(asType(parent), type);
  }
}
