/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;

import lombok.Getter;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Composite;
import nz.org.riskscape.wizard.CompositeField;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bind.CompositeBinder;

public class CompositeAskerTest extends BaseAskerTest<CompositeAsker> {

  @Composite @Getter
  public static class MyComposite {
    @CompositeField
    public String foo;

    @CompositeField
    public Long barBar;

    public String baz;

  }

   @Test
  public void asksAQuestionForEachCompositeField() {
    Question question = new Question("qqmorenoob", MyComposite.class);

    when(prompter.askWithRepeat(buildState, question.withName("qqmorenoob.foo").withType(String.class)))
      .thenReturn("some foo for you");

    when(prompter.askWithRepeat(buildState, question.withName("qqmorenoob.bar-bar").withType(Long.class)))
      .thenReturn(1L);

    ResultOrProblems<Object> unbound = subject.askUnbound(AskRequest.create(prompter, buildState, question));

    assertThat(
      unbound,
      Matchers.result(equalTo(ImmutableMap.of("foo", "some foo for you", "bar-bar", 1L)))
    );

    // should always use askWithRepeat, not ask - see comment about not bailing half way through answering
    verify(prompter, never()).ask(any(), any());

    // there's no field annotation on this, don't bind it
    verify(prompter, never()).askWithRepeat(buildState, question.withName("qqmorenoob.baz").withType(String.class));

    // for good measure, check that the composite binder can do its thing
    CompositeBinder binder = new CompositeBinder();

    assertThat(
      binder.bindValue(bindingContext, unbound.get(), MyComposite.class),
      Matchers.result(
        allOf(
          instanceOf(MyComposite.class),
          hasProperty("foo", equalTo("some foo for you")),
          hasProperty("barBar", equalTo(1L)),
          hasProperty("baz", nullValue())
        )
      )
    );
  }

  @Override
  protected CompositeAsker createSubject() {
    return new CompositeAsker(translationContext);
  }

}
