/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.Locale;

import lombok.Getter;
import nz.org.riskscape.cli.Terminal;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.i18n.MutableMessageSource;
import nz.org.riskscape.engine.i18n.TranslationContext;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepChain;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.Answer.Response;
import nz.org.riskscape.wizard.CliPrompter;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.QuestionSet;
import nz.org.riskscape.wizard.Survey;
import nz.org.riskscape.wizard.TestQuestionSet;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;
import nz.org.riskscape.wizard.bld.change.NoChange;

public abstract class BaseAskerTest<T extends BaseAsker> extends ProjectTest {

  ExpressionParser parser = ExpressionParser.INSTANCE;
  MutableMessageSource messageSource = new MutableMessageSource();
  TranslationContext translationContext =
      new TranslationContext(Locale.getDefault(), messageSource);

  T subject = createSubject();

  QuestionSet questionSet = new TestQuestionSet("mock-questions", Survey.EMPTY_SURVEY);

  Question question = new Question("question", String.class).inSet(questionSet);

  Terminal terminal = mock(Terminal.class);
  IncrementalBuildState buildState;
  CliPrompter prompter = mock(CliPrompter.class);;

  @Getter
  private Struct inputType = Struct.EMPTY_STRUCT;
  private ResultOrProblems<Response> lastResponse;


  public BaseAskerTest() {
    setInputType(Struct.EMPTY_STRUCT);

    when(prompter.getTerminal()).thenReturn(terminal);
  }

  protected abstract T createSubject();

  public Response ask() {
    this.lastResponse = subject.ask(AskRequest.create(prompter, buildState, question));
    return lastResponse.orElse(null);
  }

  /**
   * Set up the test fixtures by suppling a struct to use as the scope for any expressions or other that the question
   * might use to give the user tips.  All the other scaffolding is built around this.
   */
  public void setInputType(Struct inputTypeToSet) {
    buildState = IncrementalBuildState.empty(new IncrementalBuildState.Context(null, executionContext));
    inputType = inputTypeToSet;

    RealizedStep inputStep = RealizedStep.emptyInput("input", inputType);
    StepChain stepChain = new StepChain(Arrays.asList(inputStep.getAst()));
    PipelineDeclaration pipelineAst = PipelineDeclaration.EMPTY.add(stepChain);

    RealizedPipeline pipeline = new RealizedPipeline(
      executionContext,
      pipelineAst,
      Collections.singletonList(inputStep)
    );

    PipelineChange change =
        new NoChange(Answer.strings(new Question("start", String.class).inSet(questionSet), "ignored"));

    // append the single-step realized pipeline to the build state for the asker to find
    buildState = buildState.append(change, pipelineAst, pipeline);
  }
}
