/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;
import org.mockito.stubbing.OngoingStubbing;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Text;
import nz.org.riskscape.rl.ast.PropertyAccess;
import nz.org.riskscape.wizard.Choice;
import nz.org.riskscape.wizard.CliChoice;
import nz.org.riskscape.wizard.ExpressionHelper.AttributeMetadata;
import nz.org.riskscape.wizard.survey2.Choices;

public class AttributeAskerTest extends BaseAskerTest {

  /**
   * Every time the user is presented with choices in {@link #primeChosenValueAnswer(java.lang.String...) }
   * those choices are added to this list for later verification.
   */
  protected List<String> presentedAttributeChoices = new ArrayList<>();

  public AttributeAskerTest() {
    question = question.withType(PropertyAccess.class);

    setInputType(Struct.of(
        "exposure", Struct.of("geom", Types.GEOMETRY, "name", Types.TEXT, "value", Types.INTEGER),
        "hazard", Struct.of("geom", Types.GEOMETRY, "pga", Types.FLOATING)
    ));
  }

  @Override
  protected BaseAsker createSubject() {
    return new AttributeAsker(translationContext);
  }

  @Test
  public void canChooseAnAttributeFromInputType() {
    primeChosenValueAnswer("hazard.pga");

    assertThat(ask().getBoundValue(),
        is(parser.parse("hazard.pga")));

    assertThat(presentedAttributeChoices, contains(
        "exposure.geom", "exposure.name", "exposure.value", "exposure",
        "hazard.geom", "hazard.pga", "hazard"
    ));
  }

  @Test
  public void canChooseAnAttributeFromExpressionScope() {
    question = question.withChoices(ibs ->
      Choices.forScope(Struct.of("name", Types.TEXT, "value", Types.FLOATING), question)
    );
    primeChosenValueAnswer("value");

    assertThat(ask().getBoundValue(),
        is(parser.parse("value")));

    assertThat(presentedAttributeChoices, contains(
        "name", "value"
    ));
  }

  protected void primeChosenValueAnswer(String... valueAnswers) {
    OngoingStubbing<CliChoice> stub = when(prompter.choose(nullable(Text.class), anyList()));
    for (String answer : valueAnswers) {
      stub = stub.thenAnswer(invocation -> {
        List<CliChoice<Choice>> choices = invocation.getArgument(1);

        presentedAttributeChoices = choices.stream()
            .map(c -> c.data.getDerivedFrom(AttributeMetadata.class).getFullyQualifiedName())
            .collect(Collectors.toList());

        return choices.stream()
            .filter(choice ->
              choice.data.getDerivedFrom(AttributeMetadata.class).getFullyQualifiedName().equals(answer)
              )
            .findFirst()
            .orElseThrow(() -> new RuntimeException("choice not found"));

      });
    }
  }

}
