/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.junit.Test;

import nz.org.riskscape.engine.DefaultEngine;
import nz.org.riskscape.engine.function.maths.Min;
import nz.org.riskscape.rl.ast.StructDeclaration;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bind.AggregationExpression;
import nz.org.riskscape.wizard.bind.CompositeBinder;

/**
 * Tests the @{link AskAsAsker} with the {@link AggregationExpression}.
 */
public class AskAsAskerTest extends BaseAskerTest<AskAsAsker> {

  @Override
  public DefaultEngine createEngine() {
    engine = super.createEngine();
    engine.getBinders().add(new CompositeBinder());
    return engine;
  }

  @Test
  public void testCanBind() throws Exception {
    question = new Question("group-me", StructDeclaration.class);
    // false: missing the ask as
    assertFalse(subject.canAsk(buildState, question));

    // add the ask as and we should be able to bind now
    question = question.withAskAs(AggregationExpression.ASK_AS);
    assertTrue(subject.canAsk(buildState, question));

    assertTrue(subject.canAsk(buildState, question.withAnnotations("foo", "bar")));

  }

  @Test
  public void convertsTheThreeIndividualQuestionsInToAnExpression() {

    project.getFunctionSet().add(new Min());
    question = new Question("group-me", StructDeclaration.class).withAskAs(AggregationExpression.ASK_AS);

    when(prompter.findAsker(any(), any())).thenReturn(new CompositeAsker(translationContext));

    // the asking is done via the CompositeAsker
    expectAsk("attribute", "atta_boy");
    expectAsk("function", "min(chips)");
    expectAsk("resultName", "cool");

    // but the result is returned as a StructDeclaration
    assertEquals("min(atta_boy) as cool", ask().getOriginalInput());
  }

  void expectAsk(String fieldName, String response) {
    when(prompter.askWithRepeat(
        buildState,
        CompositeBinder.fieldifyQuestion(question, AggregationExpression.class, fieldName)
    ))
    .thenReturn(response);

  }

  @Override
  protected AskAsAsker createSubject() {
    return new AskAsAsker(translationContext);
  }

}
