/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.FunctionCallPrototype;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.wizard.Answer.Response;
import nz.org.riskscape.wizard.AskerHints;
import nz.org.riskscape.wizard.CliChoice;
import nz.org.riskscape.wizard.CliPrompter;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Text;

@SuppressWarnings("unchecked")
public class AggregationFunctionPrototypeAskerTest extends BaseAskerTest<AggregationFunctionPrototypeAsker> {

  @Override
  protected nz.org.riskscape.wizard.ask.AggregationFunctionPrototypeAsker createSubject() {
    return new AggregationFunctionPrototypeAsker(translationContext);
  }

  public AggregationFunctionPrototypeAskerTest() {
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    question = question.withType(FunctionCallPrototype.class).withAnnotations(AskerHints.AGGREGATION, "");
  }

  @Test
  public void askCanReturnSimpleAggFunction() {
    // a simple agg function is one that does not require any extra arguments
    selectWhenAsked("max");

    Response response = ask();
    assertThat(response.getOriginalInput(), is("max"));
    FunctionCallPrototype aggFunction = (FunctionCallPrototype)response.getBoundValue();
    assertThat(aggFunction.getPrototype().toSource(), is("max()"));
  }


  @Test
  public void askCanReturnAggFunctionWithExtraArgs() {
    // a simple agg function is one that does not require any extra arguments
    selectWhenAsked("percentile");

    when(prompter.readlineWithTitle((Text) any())).thenReturn("50");

    Response response = ask();
    assertThat(response.getOriginalInput(), is("percentile(x, percentile: 50)"));
    FunctionCallPrototype aggFunction = (FunctionCallPrototype)response.getBoundValue();
    assertThat(aggFunction.getPrototype().toSource(), is("percentile(x, percentile: 50)"));
  }

  @Test
  public void askCanReturnAggFunctionWithExtraListArgs() {
    // a simple agg function is one that does not require any extra arguments
    selectWhenAsked("percentiles");

    when(prompter.readlineWithTitle((Text) any())).thenReturn("50").thenReturn("75").thenReturn("90");
    when(prompter.askIf(anyString())).thenReturn(true).thenReturn(true).thenReturn(false);

    Response response = ask();
    assertThat(response.getOriginalInput(),
        is("percentiles(x, percentiles: [50, 75, 90])"));
    FunctionCallPrototype aggFunction = (FunctionCallPrototype)response.getBoundValue();
    assertThat(aggFunction.getPrototype().toSource(),
        is("percentiles(x, percentiles: [50, 75, 90])"));
  }

  @Test
  public void canAskAboutAggregation() {
    assertThat(subject.canAsk(buildState, question), is(true));
  }

  @Test
  public void cannotAskAboutOtherThings() {
    Question question = new Question("", Expression.class);
    assertThat(subject.canAsk(buildState, question), is(false));
  }

  /**
   * Primes the {@link CliPrompter} to select the option with label matching chosenId. Will also verify
   * that all options presented contain an {@link AggregationFunction}
   *
   * @param chosenId id of option to choose
   */
  void selectWhenAsked(String chosenId) {
    when(prompter.choose((Text)any(), any())).thenAnswer(invocation -> {
      List<CliChoice> options = invocation.getArgument(1, List.class);
      CliChoice chosen = null;
      for (CliChoice choice : options) {
        //All listed functions must be aggregation functions
        RiskscapeFunction function = project.getFunctionSet().get(choice.label);
        assertThat(function.getAggregationFunction().isPresent(), is(true));
        if (chosenId.equals(choice.label)) {
          chosen = choice;
        }
      }
      return chosen;
    });
  }
}
