/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import lombok.Data;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.CliPrompter;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;

/**
 * Brings together all the arguments to ask in to a single request-like object to make it easier to extend/tweak in the
 * future.
 */
public interface AskRequest {

  /**
   * Default implementation
   */
  @Data
  class Impl implements AskRequest {
    private final CliPrompter cliPrompter;
    private final IncrementalBuildState buildState;
    private final Question question;
  }

  /**
   * Create a default implementation from the given args
   */
  static AskRequest create(
      CliPrompter cliPrompter,
      IncrementalBuildState buildState,
      Question question
  ) {
    return new Impl(cliPrompter, buildState, question);
  }

  /**
   * The CLI that is used to interact with the user
   */
  CliPrompter getCliPrompter();

  /**
   * The current state of building the pipeline
   */
  IncrementalBuildState getBuildState();

  /**
   * The question being asked
   */
  Question getQuestion();

  /**
   * Bind the given response to something suitable for this ask input's question
   */
  default ResultOrProblems<Answer.Response> bind(Object raw) {
    return ResultOrProblems.of(
      new Answer.Response(
        getBuildState().getBindingContext().bind(raw, getQuestion().getParameterType()),
        raw
      )
    );
  }

  /**
   * @return the project associated with this request
   */
  default Project getProject() {
    return getBuildState().getProject();
  }

  /**
   * @return the question's parameter type
   */
  default Class<?> getParameterType() {
    return getQuestion().getParameterType();
  }

  /**
   * @return a binding context that can be used for converting user input in to engine-y objects
   */
  default BindingContext getBindingContext() {
    return getBuildState().getBindingContext();
  }

  /**
   * @return an Asker for the given question (and build state)
   */
  default Asker findAsker(IncrementalBuildState buildState, Question question) {
    return getCliPrompter().findAsker(buildState, question);
  }

}
