/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.ask;

import nz.org.riskscape.engine.i18n.TranslationContext;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.Answer.Response;
import nz.org.riskscape.wizard.AskAs;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;

/**
 * Provides a bridge between having a nice CLI UX (via the CompositeAsker) and
 * accepting any aggregate expression for the wizard questions (which simplifies
 * the config and means you can still do some advanced expression-y things with
 * a plain old wizard model).
 */
public class AskAsAsker extends BaseAsker {

  public AskAsAsker(TranslationContext translationContext) {
    super(translationContext);
  }

  @Override
  public boolean canAsk(IncrementalBuildState buildState, Question question) {
    return question.isAskAs();
  }

  @Override
  public ResultOrProblems<Response> ask(AskRequest input) {
    AskAs<?> askAs = input.getQuestion().getAskAs().get();

    Question originalQuestion = input.getQuestion();
    Question askedAsQuestion = originalQuestion.getAskedAs();
    IncrementalBuildState state = input.getBuildState();

    Asker delegatedAsker = input.findAsker(state, askedAsQuestion);

    // offload all the asking work to delegated asker
    ResultOrProblems<Object> unboundOr = delegatedAsker.askUnbound(
        AskRequest.create(input.getCliPrompter(), state, askedAsQuestion));

    return unboundOr
      // first bind to askedAsQuestions parameter type
      .flatMap(unbound -> Answer.bind(state.getBindingContext(), askedAsQuestion, unbound))
      // now use AskAs.convert to bind to originalQuestions parameter type
      .flatMap(response ->  askAs.convert(state.getBindingContext(), originalQuestion, response));
  }
}
