/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import static nz.org.riskscape.wizard.ExpressionHelper.*;

import java.util.List;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.cli.ApplicationCommand;
import nz.org.riskscape.engine.i18n.TranslationContext;
import nz.org.riskscape.picocli.CommandLine.Command;
import nz.org.riskscape.wizard.ask.AskAsAsker;
import nz.org.riskscape.wizard.ask.AggregationFunctionPrototypeAsker;
import nz.org.riskscape.wizard.ask.Asker;
import nz.org.riskscape.wizard.ask.AttributeAsker;
import nz.org.riskscape.wizard.ask.BooleanExpressionAsker;
import nz.org.riskscape.wizard.ask.ChoiceAsker;
import nz.org.riskscape.wizard.ask.CompositeAsker;
import nz.org.riskscape.wizard.ask.DefaultAsker;
import nz.org.riskscape.wizard.ask.ExpressionAsker;
import nz.org.riskscape.wizard.model2.ModelSurvey2;

@Command(name = "wizard")
public class WizardCommand extends ApplicationCommand {

  @Override
  public Object doCommand(Project project) {
    TranslationContext askerMessageSource =
        new TranslationContext(getTerminal().getLocale(), getMessages().newMessageSource("wizard"));

    CliPrompter prompter = new CliPrompter(
            getTerminal(),
            getAskers(askerMessageSource),
            new DefaultAsker(askerMessageSource),
            getMessages(),
            askerMessageSource
    );

    WizardActions wizardActions = new WizardActions(prompter, project.getRelativeTo());

    Survey survey = new ModelSurvey2(
        getMessages().newMessageSource(Survey.getSurveyMessageBundleName("survey2"))
    );

    CliWizardProcessor2 processor = new CliWizardProcessor2(project, prompter, wizardActions);
    processor.run(survey);

    return null;
  }

  public static List<Asker> getAskers(TranslationContext askerMessageSource) {
    return List.of(
      new AggregationFunctionPrototypeAsker(askerMessageSource),
      // BooleanExpressionAsker needs to be above AskAsAnswer so we can have the askAs behaviour on the platform,
      // but keep the old asker behaviour here
      new BooleanExpressionAsker(askerMessageSource),
      new AskAsAsker(askerMessageSource),
      new ChoiceAsker<Boolean>(askerMessageSource, Boolean.class, q -> CliPrompter.YES_NO_CHOICES),
      new AttributeAsker(askerMessageSource),
      new ExpressionAsker(askerMessageSource, Lists.newArrayList(TAG_EXPRESSION_TYPE_NUMERIC)),
      new CompositeAsker(askerMessageSource)
    );

  }
}
