/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.postgis;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.util.Map;
import java.util.Optional;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.problem.GeneralProblems;

public class PostGISResolverTest extends ProjectTest {

  BindingContext context = super.bindingContext;
  PostGISResolver resolver = new PostGISResolver(engine);
  PostGISResolver.Params params = new PostGISResolver.Params();
  Bookmark bookmark;

  @Test
  public void canCreateFromAURI() {
    URI location = URI.create("postgis://foo-user:password@cool-host.internet.com:1200/foo");
    bookmark = new Bookmark("foo", "foos", null, location, Maps.newHashMap());

    bindAndValidate();

    Map<String, Object> expected = ImmutableMap.<String, Object>builder()
        .put("user", "foo-user")
        .put("passwd", "password")
        .put("host", "cool-host.internet.com")
        .put("port", 1200)
        .put("database", "foo")
        .put("dbtype", "postgis")
        .build();
    assertEquals(expected, params.storeParams);
  }

  @Test
  public void canCreateFromMinimalURI() {
    URI location = URI.create("postgis://petey@localhost/foo");
    bookmark = new Bookmark("foo", "foos", null, location, Maps.newHashMap());

    bindAndValidate();

    Map<String, Object> expected = ImmutableMap.<String, Object>builder()
        .put("user", "petey")
        .put("host", "localhost")
        .put("port", 5432)
        .put("database", "foo")
        .put("dbtype", "postgis")
        .build();

    assertEquals(expected, params.storeParams);
  }

  @Test
  public void missingRequiredParametersAreNaughty() throws Exception {
    URI location = URI.create("postgis://localhost");
    bookmark = new Bookmark("foo", "foos", null, location, Maps.newHashMap());

    bindAndValidate();

    assertThat(params.problems, containsInAnyOrder(
      GeneralProblems.get().required("user"),
      GeneralProblems.get().required("database")
    ));
  }

  @Test
  public void specificParametersTakePrecedenceOverUriOnes() throws Exception {
    URI location = URI.create("postgis://foo:cool@localhost:5432/foo");
    bookmark = Bookmark.builder().id("foo").location(location).build();

    params.user = Optional.of("rad");
    params.database = Optional.of("database-override");
    params.port = Optional.of(1000);
    params.password = Optional.of("secret");

    bindAndValidate();

    Map<String, Object> expected = ImmutableMap.<String, Object>builder()
        .put("user", "rad")
        .put("passwd", "secret")
        .put("host", "localhost")
        .put("port", 1000)
        .put("database", "database-override")
        .put("dbtype", "postgis")
        .build();
    assertEquals(expected, params.storeParams);
  }

  public void bindAndValidate() {

    params.bookmark = bookmark;
    params.bindingContext = context;

    resolver.validateParameters(params, context);
  }
}
