/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.postgis;

import java.net.URI;

import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * PostGIS related problems.
 */
public interface PostGISProblems extends ProblemFactory {

  Problem invalidLocation(URI location);

  /**
   * A connection failure problem
   * @param location  the PostGIS location that failed to connect
   * @param failureReason the reason for the failure
   */
  Problem connectionFailure(URI location, String failureReason);

  /**
   * @return a tip for use with {@link #connectionFailure(java.net.URI, java.lang.String) }
   */
  @SeverityLevel(Problem.Severity.INFO)
  Problem connectionTipSpelling();

  /**
   * @return a tip for use with {@link #connectionFailure(java.net.URI, java.lang.String) }
   */
  @SeverityLevel(Problem.Severity.INFO)
  Problem connectionTipPassword();

  /**
   * @return a tip for use with {@link #connectionFailure(java.net.URI, java.lang.String) }
   */
  @SeverityLevel(Problem.Severity.INFO)
  Problem connectionTipConnectivity();

  /**
   * When spatial data is being written but the target database does not have the postgis extension enabled.
   */
  Problem databaseNotSpatial();

  /**
   * A password was found in clear text, either in the location or a bookmark parameter.
   *
   * This is insecure, user should put the password in the PGPASSWORD environment variable.
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem passwordInClearText();

  /**
   * The location contained excess parts that will be ignored.
   *
   * @param ignored the part of the URI that is being ignored
   * @param from    identifies what part of the URI ignored is from, eg, query, fragment etc
   */
  @SeverityLevel(Problem.Severity.WARNING)
  Problem ignoredFromLocation(String ignored, String from);

  static PostGISProblems get() {
    return Problems.get(PostGISProblems.class);
  }
}
