/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.oq.sitecol;

import java.nio.ByteBuffer;
import java.util.Iterator;

import nz.org.riskscape.hdf5.types.H5CompoundType;
import nz.org.riskscape.hdf5.H5Dataset;
import nz.org.riskscape.hdf5.cursor.H5FixedSizeCursor;

/**
 * Efficient and correct iteration across the 'sitecol' dataset, yielding just the site id and coordinates.
 */
public class SitecolIterator implements Iterator<Site> {

  private long index = 0;

  private final H5FixedSizeCursor cursor;
  private H5CompoundType h5Type;
  private final int lonIndex;
  private final int latIndex;

  public SitecolIterator(H5Dataset dataset) {
    this(dataset, "lon", "lat");
  }

  public SitecolIterator(H5Dataset dataset, String lonName, String latName) {
    this.cursor = (H5FixedSizeCursor) dataset.openCursor();
    this.h5Type = (H5CompoundType) dataset.getDataType();
    this.lonIndex = h5Type.findIndex(lonName);
    this.latIndex = h5Type.findIndex(latName);
  }

  @Override
  public boolean hasNext() {
    return cursor.hasNext();
  }

  @Override
  public Site next() {
    ByteBuffer bb = cursor.getByteBuffer();

    Number latitude = (Number)cursor.peek(h5Type.getMembers().get(latIndex));

    Number longitude = (Number) cursor.peek(h5Type.getMembers().get(lonIndex));

    Site site = new Site(latitude.floatValue(), longitude.floatValue(), index);

    index++;
    cursor.setCurrentIndex(index);

    return site;
  }

}
