/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.ogr.tests;

import static nz.org.riskscape.engine.GeoHelper.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.hamcrest.Matchers;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;

import com.google.common.collect.ImmutableMap;

import nz.org.riskscape.engine.GeoHelper;
import nz.org.riskscape.engine.cli.BaseCliIntegrationTest;
import nz.org.riskscape.engine.cli.CliIntegrationTest;
import nz.org.riskscape.engine.cli.PostgisIntegrationTestRunner;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.test.SkipTestOnWindows;

@RunWith(PostgisIntegrationTestRunner.class)
@Category(CliIntegrationTest.class)
public class RM316_Postgis_Via_Ogr extends BaseCliIntegrationTest {

  Struct stationType = Struct.of("name", Nullable.TEXT);
  Struct outputType = Struct.of("output", stationType);

  @Test
  @Category(SkipTestOnWindows.class)
  public void canUseRelationFromPostgis() throws Exception {

    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("model")
        .add("run")
        .add("stations")
        .add("--output=output")
        .add("--format=csv"));

    assertEquals(Integer.valueOf(0), result.retCode);

    assertThat(readColumnsFromCsv(outputDir.resolve("output.csv"), "t50_fid", "name", "geom"), Matchers.allOf(
        hasItem(contains(
            is("6082054"),
            is("Greenlane Station"),
            GeoHelper.wktGeometryMatch("POINT (-36.88965260770787 174.79744423074914)", DEGREE_TOLERANCE_NEAREST_MM)
        )),
        hasItem(contains(
            is("6082051"),
            is("Penrose Station"),
            GeoHelper.wktGeometryMatch("POINT (-36.91012202505844 174.8157244628168)", DEGREE_TOLERANCE_NEAREST_MM)
        )),
        hasItem(contains(
            is("6082053"),
            is("Ellerslie Station"),
            GeoHelper.wktGeometryMatch("POINT (-36.898725475187774 174.80824125198643)", DEGREE_TOLERANCE_NEAREST_MM)
        ))
    ));
  }

  @Ignore // the results are clearly incorrect when the reproject to lat/long is included. GL#749 for this issue.
  @Test
  @Category(SkipTestOnWindows.class)
  public void rm315GDBviaOGR() throws Exception {

    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("model")
        .add("run")
        .add("stations")
        .add("-p=stations='nz_rail_stations_gdb'")
        .add("--output=output")
        .add("--format=csv"));

    assertEquals(Integer.valueOf(0), result.retCode);

    Map<String, List<String>> expected = ImmutableMap.of(
        "6082054", Arrays.asList("Greenlane Station", "POINT (-36.88965260770787 174.79744423074914)"),
        "6082051", Arrays.asList("Penrose Station", "POINT (-36.91012202505844 174.8157244628168)"),
        "6082053", Arrays.asList("Ellerslie Station", "POINT (-36.898725475187774 174.80824125198643)")
    );
    assertHasRecords(expected, outputDir.resolve("output.csv"), "t50_fid", "name", "geom");
  }

}
