/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ogr;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.File;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.BookmarkFactory;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;


public class GdbResolverTest extends ProjectTest {

  BindingContext context = super.bindingContext;
  GdbResolver resolver = new GdbResolver(engine, new OgrHelper());

  @Before
  public void setup() {
    engine.getBookmarkResolvers().add(resolver);
  }

  @Test
  public void testFilesEndingInGdbAreResolved() {
    Bookmark bookmark = new Bookmark("foo", "bar", null, URI.create("file:///foo/bar.gdb/"), Collections.emptyMap());
    assertTrue(resolver.resolve(bookmark, context).isPresent());
  }

  @Test
  public void errorGivenIfUriDoesNotPointToALocalDirectory() throws Exception {
    File tmpdir = File.createTempFile("test", ".gdb");
    Bookmark bookmark = new Bookmark("foo", "bar", null, tmpdir.toURI(), Collections.emptyMap());
    assertTrue(resolver.resolve(bookmark, context).isPresent());
    ResolvedBookmark rb = resolver.resolve(bookmark, context).get();

    // ci env doesn't necessarily have ogr installed, so we ca0n't use all-or-nothing contains matcher
    assertThat(rb.validate(), hasItem(isProblem(Severity.ERROR, containsString("not a directory"))));
  }

  @Test
  public void canPutLayerInAFileUri() throws Exception {
    Path gdbFile = Paths.get("src", "test", "resources", "nz", "org", "riskscape", "ogr", "empty.gdb");
    String pathWithParams = gdbFile.toString() + "?layer=non-existant";
    URI uri = BookmarkFactory.uriFromLocation(pathWithParams, new File(".").getAbsoluteFile().toURI()).get();
    Bookmark bookmark = Bookmark.fromURI(uri);


    ResolvedBookmark resolved = resolver.resolve(bookmark, context).get();
    assertThat(Problem.hasErrors(resolved.validate()), is(false));  //There will be no ogr warning on systems that do
                                                                    //not have gdal installed
  }

}
