/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ogr;

import java.io.IOException;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.geotools.data.ogr.OGRDataStore;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.data.relation.FeatureSourceBookmarkResolver;
import nz.org.riskscape.engine.data.relation.RelationBookmarkParams;
import nz.org.riskscape.problem.Problem;


public class OgrResolver extends FeatureSourceBookmarkResolver<OGRDataStore, OgrResolver.OgrParams> {

  public static class OgrParams extends RelationBookmarkParams {

    @ParameterField()
    public String ogrDriver;

    @ParameterField()
    public Optional<String> ogrSource = Optional.empty();
  }

  private final OgrHelper ogrHelper;

  public OgrResolver(Engine engine, OgrHelper helper) {
    super(engine);
    this.ogrHelper = helper;
  }

  protected Map<String, Serializable> createOgrConnectionParameters(OgrParams params) {
    if (!params.ogrSource.isPresent()) {
      Path ogr = getBookmarkedPath(params);
      params.ogrSource = Optional.of(ogr.toFile().toString());
    }

    return ImmutableMap.of(
        "DriverName", params.ogrDriver,
        "DatasourceName", params.ogrSource.get());
  }

  @Override
  protected OGRDataStore createDataStore(OgrParams params) throws MalformedURLException, IOException {
     OGRDataStore ds = (OGRDataStore) ogrHelper.getDataSourceFactory()
          .orElseThrow(() -> new RiskscapeException("OGR not available on this system"))
          .createDataStore(createOgrConnectionParameters(params));
      return ds;
  }

  @Override
  protected void validateParameters(OgrParams params, BindingContext context) {

    if (!ogrHelper.isAvailable()) {
      params.problems.add(Problem.warning("OGR does not appear to be installed on this system"));
    } else if (!ogrHelper.getAvailableDrivers().contains(params.ogrDriver)) {
        params.problems.add(Problem.error(
            "ogr driver %s is not available on this system. Run 'ogrinfo --formats' to list available drivers",
            params.ogrDriver));

    }
  }
  @Override
  public Set<String> getFormats() {
    return Sets.newHashSet("ogr");
  }

  @Override
  protected Map<String, String> getExtensionsToFormats() {
    return Collections.emptyMap();
  }

}
