/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.netcdf;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.cli.tests.BaseModelRunCommandTest;
import nz.org.riskscape.engine.test.EngineTestPlugins;

/**
 * Note that some of these tests can be slow to run because they download NetCDF files from Unidata.
 */
@EngineTestPlugins({"defaults", "netcdf"})
public class NetCDFexamplesTest extends BaseModelRunCommandTest {

  @Override
  public Path stdhome() {
    return Paths.get("..", "..", "examples", "netcdf");
  }

  @Test
  public void canRunAtmosphericModel() throws Exception {
    List<List<String>> results = runModel("atmospheric", "exposure.name", "sampled.pr", "sampled.tas");
    assertThat(results, containsInAnyOrder(
        Arrays.asList("Auckland", "3.822290091193281E-5", "14.609002685546898"),
        Arrays.asList("Wellington", "4.5984474127180874E-5", "11.069696044921898"),
        Arrays.asList("Christchurch", "3.6739642382599413E-5", "9.416680908203148"),
        Arrays.asList("Dunedin", "5.4601143347099423E-5", "7.253411865234398")
    ));
  }

  @Test
  public void canRunSeaTempByTime() throws Exception {
    List<List<String>> results = runModel("sea-temperature", "exposure.name", "readings.time", "readings.tos");
    // spot check some results
    assertThat(results, hasItems(
        Arrays.asList("Wellington", "465.0", "11.632989501953148"),
        Arrays.asList("Wellington", "435.0", "13.246575927734398")
    ));
  }

  @Test
  public void canRunSeaTempAggregatedModel() throws Exception {
    List<List<String>> results = runModel("sea-temperature-aggregated", "exposure.name",
        "readings.min_tos", "readings.max_tos");
    assertThat(results, containsInAnyOrder(
        Arrays.asList("Auckland", "9.628686523437523", "15.541009521484398"),
        Arrays.asList("Wellington", "7.754937744140648", "14.440515136718773"),
        Arrays.asList("Christchurch", "7.309320068359398", "12.989709472656273"),
        Arrays.asList("Dunedin", "7.160028076171898", "11.834619140625023")
    ));
  }

  @Test
  public void canRunScalableNetCDFExample() throws Exception {
    List<List<String>> results = runModel("scalable-netcdf", "exposures.name", "sea_temp.time", "sea_temp.tos");
    // same results spot check as for the sea-temperature example
    assertThat(results, hasItems(
        Arrays.asList("Wellington", "465.0", "11.632989501953148"),
        Arrays.asList("Wellington", "435.0", "13.246575927734398")));
  }

  List<List<String>> runModel(String modelId, String... expectedColumns) throws Exception {
    runCommand.modelId = modelId;
    runCommand.runnerOptions.format = "csv";
    runCommand.run();

    return readCsvColumns("results.csv", expectedColumns);
  }

}
