/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.netcdf;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.cli.tests.BaseModelRunCommandTest;
import nz.org.riskscape.engine.test.EngineTestPlugins;

@EngineTestPlugins({"defaults", "netcdf"})
public class NetCDFIntegrationTest extends BaseModelRunCommandTest {

  @Test
  public void canReadDimensionData() throws Exception {
    // this test will read the LON variable, which is a 1D variable used mainly as a dimension
    // for other data
    List<List<String>> results = readNetCdfBookmark("test-lon", "LON");

    assertThat(results, contains(
        Arrays.asList("174.1"),
        Arrays.asList("174.2"),
        Arrays.asList("174.3"),
        Arrays.asList("174.4")
    ));

  }

  @Test
  public void canRead1Ddata() throws Exception {
    // this test will read the EVENT variable, which is a 1D variable whose dimension is EVENTID.
    // the eventid dimension is difference to other dimensions like lon because it has no variable
    // data. In this case the dimension index is used as the dimension value.
    List<List<String>> results = readNetCdfBookmark("test-event", "EVENTID", "EVENT");

    assertThat(results, contains(
        Arrays.asList("0", "event1"),
        Arrays.asList("1", "event2"),
        Arrays.asList("2", "event3"),
        Arrays.asList("3", "event4")
    ));

  }

  @Test
  public void canRead2Ddata() throws Exception {
    // reads the 2D site data.
    List<List<String>> results = readNetCdfBookmark("test-site", "LAT", "LON", "SITE");

    assertThat(results, contains(
        Arrays.asList("-41.1", "174.1", "s11"),
        Arrays.asList("-41.2", "174.1", "s21"),
        Arrays.asList("-41.3", "174.1", "s31"),
        Arrays.asList("-41.4", "174.1", "s41"),
        Arrays.asList("-41.1", "174.2", "s12"),
        Arrays.asList("-41.2", "174.2", "s22"),
        Arrays.asList("-41.3", "174.2", "s32"),
        Arrays.asList("-41.4", "174.2", "s42"),
        Arrays.asList("-41.1", "174.3", "s13"),
        Arrays.asList("-41.2", "174.3", "s23"),
        Arrays.asList("-41.3", "174.3", "s33"),
        Arrays.asList("-41.4", "174.3", "s43"),
        Arrays.asList("-41.1", "174.4", "s14"),
        Arrays.asList("-41.2", "174.4", "s24"),
        Arrays.asList("-41.3", "174.4", "s34"),
        Arrays.asList("-41.4", "174.4", "s44")
    ));
  }

  @Test
  public void canRead3Ddata() throws Exception {
    List<List<String>> results = readNetCdfBookmark("test-data", "TIME", "LAT", "LON", "DATA1", "DATA2");

    assertThat(results, hasSize(32));
    assertThat(results, hasItems(
        // spot check some expected results
        Arrays.asList("1.0", "-41.1", "174.1", "111.0", "311.0"),
        Arrays.asList("1.0", "-41.1", "174.3", "113.0", "313.0"),
        Arrays.asList("2.0", "-41.4", "174.4", "244.0", "444.0")
    ));
  }

  @Test
  public void canReadRandomLayer() throws Exception {
    // this bookmark doesn't specify what layer to include so the first variable from the netcdf is choosen
    // for us. This seems to be the data layer, but I guess if the libs change then this test could break to.
    // for now though it looks like we read DATA2
    List<List<String>> results = readNetCdfBookmark("test-random-layer", "TIME", "LAT", "LON", "DATA2");

    assertThat(results, hasSize(32));
    assertThat(results, hasItems(
        // spot check some expected results
        Arrays.asList("1.0", "-41.1", "174.1", "311.0"),
        Arrays.asList("1.0", "-41.1", "174.3", "313.0"),
        Arrays.asList("2.0", "-41.4", "174.4", "444.0")
    ));
  }

  @Test
  public void canReadJoinedData() throws Exception {
    runCommand.modelId = "read-data-site";
    runCommand.run();

    List<List<String>> results = openCsv("output.csv", "TIME", "LAT", "LON", "DATA1", "DATA2", "SITE");

    assertThat(results, hasSize(32));
    assertThat(results, hasItems(
        // spot check some expected results
        Arrays.asList("1.0", "-41.1", "174.1", "111.0", "311.0", "s11"),
        Arrays.asList("1.0", "-41.1", "174.3", "113.0", "313.0", "s13"),
        Arrays.asList("2.0", "-41.4", "174.4", "244.0", "444.0", "s44")
    ));
  }

  @Test
  public void errorIfIncludedLayerNotKnown() throws Exception {
    RiskscapeException e =
        Assert.assertThrows(RiskscapeException.class, () -> readNetCdfBookmark("test-unknown-dimension"));

    assertThat(render(e.getProblem()), containsString(
        "'UNKNOWN' is not a valid option for layer. "
            + "Available options are: [DATA2, DATA1, EVENT, SITE, LON, TIME, LAT]"));

  }

  @Test
  public void errorIfIncludedLayersHaveDifferentDimensions() {
    RiskscapeException e =
        Assert.assertThrows(RiskscapeException.class, () -> readNetCdfBookmark("test-mixed-dimensions"));

    assertThat(render(e.getProblem()), containsString(
        "One of the layers you have chosen, DATA1, has different dimensions ([LAT, LON]) "
            + "from the others ([TIME, LAT, LON])."
    ));
  }

  List<List<String>> readNetCdfBookmark(String bookmark, String... expectedColumns) throws Exception {
    runCommand.modelId = "read-all";
    runCommand.parameters.add(String.format("input='%s'", bookmark));

    runCommand.run();

    return openCsv("output.csv", expectedColumns);
  }

}
