/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.netcdf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.relation.BaseRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

import ucar.ma2.Array;
import ucar.nc2.Dimension;
import ucar.nc2.Variable;
import ucar.nc2.dataset.NetcdfDataset;

/**
 * A relation created from one or more layers in a NetCDF file.
 *
 * NetCDF files contain variables (similar to layers) that are indexed by one or more dimensions. Typical
 * dimensions are lat, lon and time. Other dimensions are possible as well.
 *
 * In many cases there will be a variable with the same name as the dimension that it is indexed by. These
 * are actually the values of the dimension. Dimensions that do not have an associated variable the dimension
 * data is the index.
 *
 * For example given:
 * lat(lat): 20, 30
 * lon(lon): 50, 60
 * data(lat, lon, t): 10, 11, 12, 20, 21, 22, 30, 31, 32
 *
 * Then data(20, 50, 0) = 10, data(30, 50, 1) = 11 etc.
 */
public class NetCdfRelation extends BaseRelation {

  public interface LocalProblems extends ProblemFactory {

    Problem mismatch(String layer, List<String> required, List<String> received);
  }

  /**
   * An {@link Iterator} that for each item in the dataset produces the dimension indexes to access
   * that item.
   *
   * For example, a two dimensional dataset would look a little like a chess board. This iterator will
   * work across the top row, then the second etc.
   *
   * Package scoped for test access only.
   */
  static class IndexIncrementer implements Iterator<int[]> {

    private final List<Dimension> dimensions;
    private final int[] indexes;

    IndexIncrementer(List<Dimension> dimensions) {
      this.dimensions = dimensions;
      this.indexes = new int[dimensions.size()];
      this.indexes[0] = -1;
    }

    @Override
    public boolean hasNext() {
      for (int i = 0; i < dimensions.size(); i++) {
        if (indexes[i] < dimensions.get(i).getLength() - 1) {
          // we have a next value when one of the indexes can be incremented without going past it's end.
          return true;
        }
      }
      return false;
    }

    @Override
    public int[] next() {
      // first we need to increment the indexes
      for (int i = 0; i < dimensions.size(); i++) {
        if (indexes[i] < dimensions.get(i).getLength() - 1) {
          ++indexes[i];
          // now that this index has been incremented (but not past it's end) then we are done.
          break;
        }
        // if we set this index back to zero then we need to continue the loop to increment the
        // next index
        indexes[i] = 0;
      }

      // now just return what we have
      return indexes;
    }

  }

  static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  /**
   * Create a {@link Relation} from the given layers in the NetCDF dataset.
   *
   * @param ncd NetCDF dataset
   * @param layers  layers (variables) to include in relation
   * @return relation or problems encountered
   */
  public static ResultOrProblems<Relation> of(NetcdfDataset ncd, List<String> layers) {
    // a list of the variables that will be included in the results
    List<Variable> collectedVariables = new ArrayList<>();
    // a list of dimensions that index the included variables. All variables should be indexed by
    // the same dimensions.
    List<Dimension> collectedDimensions = null;

    // we need to check that each layer exists and is compatible with the others
    for (String layer : layers) {
      Variable var = ncd.findVariable(layer);

      // List of dimensions that index this variable. We need to make sure these are consistent
      // with #collectedVariables.
      List<Dimension> varDimensions;

      if (var != null) {
        varDimensions = var.getDimensions();

        // we need to add the layer variable to the collected variables.
        // but there is a special case when the user has requested a layer that is one of the dimension
        // variables. In NetCDF's the variables for dimensions are defined to be indexed by themselves
        // eg `lon(lon)`. There is only one set of data so in this case we don't add the variable to
        // collected variables as the same thing will be in collectedDimensions (other code will die if
        // there are no dimensions hence using dimensions rather than variables).
        if (! (varDimensions.size() == 1 && varDimensions.get(0).getShortName().equals(var.getShortName()))) {
          collectedVariables.add(var);
        }
      } else {
        // the layer does not exist
        List<String> available = ncd.getVariables().stream()
            .map(Variable::getShortName)
            .collect(Collectors.toList());
        return ResultOrProblems.failed(GeneralProblems.get().notAnOption(layer, "layer", available));
      }

      if (collectedDimensions == null) {
        // first layer encountered, the varDimensions are the dimensions to use
        collectedDimensions = varDimensions;
      } else if (!collectedDimensions.equals(varDimensions)) {
        // this layer has a different list of dimensions.
        List<String> requiredDims = collectedDimensions.stream()
            .map(Dimension::getShortName)
            .collect(Collectors.toList());

        List<String> receivedDims = varDimensions.stream()
            .map(Dimension::getShortName)
            .collect(Collectors.toList());
        return ResultOrProblems.failed(PROBLEMS.mismatch(layer, requiredDims, receivedDims));
      }
    }

    // Now we build the resulting struct. It will contain the dimensions, then each layer variable
    StructBuilder sb = Struct.builder();
    for (Dimension dimension : collectedDimensions) {
      Variable dimVariable = ncd.findVariable(dimension.getShortName());
      Type type = Types.INTEGER;
      if (dimVariable != null) {
        type = Types.fromJavaType(dimVariable.getDataType().getPrimitiveClassType());
      }
      sb.add(dimension.getShortName(), type);
    }
    for (Variable variable : collectedVariables) {
      Type type = Types.fromJavaType(variable.getDataType().getPrimitiveClassType());
      sb.add(variable.getShortName(), type);
    }
    return ResultOrProblems.of(new NetCdfRelation(ncd, collectedVariables, collectedDimensions, sb.build()));
  }

  private final NetcdfDataset ncd;

  /**
   * The NetCDF {@link Dimension}s that are used to index the {@link #variables}.
   */
  private final List<Dimension> dimensions;

  /**
   * The NetCDF {@link Variable}s to return. Along with the {@link #dimensions} these form the data
   * that is returned by the relation.
   */
  private final List<Variable> variables;

  private NetCdfRelation(NetcdfDataset ncd, List<Variable> variables, List<Dimension> dimensions,
      Struct produced) {
    super(produced);
    this.ncd = ncd;
    this.variables = variables;
    this.dimensions = dimensions;
  }

  private NetCdfRelation(NetcdfDataset ncd, List<Variable> variables, List<Dimension> dimensions,
      Fields fields) {
    super(fields);
    this.ncd = ncd;
    this.variables = variables;
    this.dimensions = dimensions;
  }

  @Override
  protected TupleIterator rawIterator() {
    try {
      Array[] data;
      Array[] dimensionData;
      synchronized (ncd) {
      // we need to fetch the variables we're going to iterate over
      data = new Array[variables.size()];
      for (int i = 0; i < data.length; i++) {
        data[i] = variables.get(i).read();
      }

      // we also need the dimension data as that's included in the returned data
      dimensionData = new Array[dimensions.size()];
      for (int i = 0; i < dimensions.size(); i++) {
        Variable variable = ncd.findVariable(dimensions.get(i).getShortName());
        if (variable != null) {
          dimensionData[i] = variable.read();
        }
      }
      }

      return TupleIterator.wrappedAndMapped(new IndexIncrementer(dimensions), idx -> {
        Object[] values = new Object[getRawType().size()];

        // first up we populate the dimension data. Dimension data comes before variable data
        // in the returned tuple.
        for (int i = 0; i < dimensions.size(); i++) {
          Type memberType = getRawType().getMembers().get(i).getType();
          // dimension data is always 1-d
          Array variableData = dimensionData[i];
          if (variableData == null) {
            values[i] = idx[i];
          } else {
            values[i] = memberType.coerce(dimensionData[i].getObject(idx[i]));
          }
        }

        // now we populate the variable(s) data.
        for (int i = 0; i < data.length; i++) {
          Type memberType = getRawType().getMembers().get(i + dimensions.size()).getType();
          Array dataArray = data[i];
          values[i + dimensions.size()] = memberType.coerce(dataArray.getObject(dataArray.getIndex().set(idx)));
        }
        return Tuple.ofValues(getRawType(), values);
      });
    } catch (IOException e) {
      throw new RiskscapeException(Problems.caught(e));
    }
  }

  @Override
  protected BaseRelation clone(Fields fields) {
    return new NetCdfRelation(ncd, variables, dimensions, fields);
  }

  @Override
  public String getSourceInformation() {
    return ncd.toString();
  }

}
