/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.jython;

import static org.junit.Assert.*;

import java.io.IOException;
import java.net.URISyntaxException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.FunctionSet;
import nz.org.riskscape.engine.PluginProjectTest;
import nz.org.riskscape.engine.function.FunctionMetadata;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.StringResource;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.problem.ResultOrProblems;

public class JythonFactoryTest extends PluginProjectTest {

  TypeSet typeSet;
  JythonFactory factory;
  FunctionSet functionSet;

  @Before
  public void createFactory() {
    typeSet = project.getTypeSet();
    functionSet = project.getFunctionSet();
    factory = new JythonFactory(project);
  }

  @After
  public void closeFactory() throws IOException {
    factory.close();
  }

  public ResultOrProblems<IdentifiedFunction> createFromString(String script) {
    try {
      Resource resource = new StringResource("test.py", script);
      return factory.buildFunction(factory.getFunctionID(resource), resource);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

  public ResultOrProblems<IdentifiedFunction> createFromString(String script, FunctionMetadata metadata) {
    try {
      Resource resource = new StringResource("test.py", script);
      return factory.buildFunction(resource, metadata);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

  @Test
  public void canPeekFunctionId() throws Exception {
    // with single quotes
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=\'my-function\'")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "   ID=\'my-function\'")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID   =\'my-function\'")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=   \'my-function\'")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=\'my-function\'   ")));

    // with double quotes
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=\"my-function\"")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "   ID=\"my-function\"")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID   =\"my-function\"")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=   \"my-function\"")));
    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown", "ID=\"my-function\"   ")));


    assertEquals("my-\'function", factory.getFunctionID(new StringResource("unknown", "ID=\'my-\'function\'")));
    assertEquals("my-\"function", factory.getFunctionID(new StringResource("unknown", "ID=\"my-\"function\"   ")));

    assertEquals("my-function", factory.getFunctionID(new StringResource("unknown",
        "Some preceeding lines",
        "ID=\"my-function\"",
        "Some trailling lines"
    )));
  }
}
