/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.jython;

import java.util.List;

import org.python.core.Py;
import org.python.core.PyException;
import org.python.core.PyFunction;
import org.python.core.PyObject;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionCallException;
import nz.org.riskscape.engine.function.JavaFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

@RequiredArgsConstructor
public class JythonRiskscapeFunctionDelegate implements JavaFunction.Delegate, JavaJythonHelpers {

  private final PyFunction function;
  private final String id;
  @Getter
  private final Type returnType;

  @Getter
  private final ArgumentList arguments;

  public interface LocalProblems extends ProblemFactory {
    Problem executionError(String functionName);
  }

  @Override
  public Object apply(List<Object> args) {
    PyObject functionResult = null;
    try {
      PyObject[] pyArgs = new PyObject[args.size()];
      for (int i = 0; i < args.size(); i++) {
        Object arg = args.get(i);
        // TODO this isn't exhaustive - lists will need to be visited as well?
        if (arg instanceof Tuple t) {
          arg = t.toMap();
        }
        pyArgs[i] = Py.java2py(arg);
      }

      functionResult = function.__call__(pyArgs);
    } catch (PyException t) {
      // anything thrown from the function is bound to be a poorly written function. So we wrap the
      //cause up in a FunctionCallExceptions so an acceptable error message is output.
      throw new FunctionCallException(
          Problems.get(LocalProblems.class).executionError(id)
            .withChildren(JythonScriptException.toProblem(t)),
          t);
    }

    if (functionResult == null) {
      throw new RuntimeException(String.format("Python function %s did not return a result", id));
    }

    return returnType.coerce(convertFromPython(functionResult, getReturnType()));
  }

}
