/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.jython;

import java.util.List;

import org.python.core.PyFunction;

import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionMetadata;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.JavaFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.UserDefinedFunction;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Realizes the Jython function against the given arguments. This is mostly so
 * that CPython (i.e. {@link CPythonRealizableFunction} and Jython behave the
 * same way, WRT to things like error handling.
 */
public class JythonRealizableFunction extends UserDefinedFunction {

  private final PyFunction pythonFunction;

  private IdentifiedFunction identified;

  public JythonRealizableFunction(FunctionMetadata details, Resource pythonScript, PyFunction pythonFunction) {
    super(details, pythonScript);
    this.pythonFunction = pythonFunction;
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) {
    if (areArgumentsCompatible(context, argumentTypes)) {
      return ResultOrProblems.of(buildFunction(ArgumentList.anonymous(argumentTypes)));
    } else {
      return ResultOrProblems.failed(getArgumentProblems(context, argumentTypes));
    }
  }

  private IdentifiedFunction buildFunction(ArgumentList args) {
    JythonRiskscapeFunctionDelegate delegate = new JythonRiskscapeFunctionDelegate(pythonFunction,
        getMetadata().getId(), getMetadata().getReturnType(), args);
    return new JavaFunction(getMetadata(), delegate);
  }

  @Override
  public IdentifiedFunction identified() {
    if (identified == null) {
      // we let the (realizable) Jython function be called directly. This can happen if a user calls
      // another user-defined function from within their Jython code. We just realize the function
      // against the expected arguments and assume that the user is passing the right data types through
      IdentifiedFunction realized = buildFunction(getMetadata().getArguments());
      identified = new MetadataIdentifiedRealizableFunction(this, getMetadata()) {
        @Override
        public Object call(List<Object> args) {
          return realized.call(args);
        }
      };
    }
    return identified;
  }
}
