/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.jython;

import java.util.List;
import java.util.Map;

import lombok.Getter;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.function.FunctionFramework;
import nz.org.riskscape.engine.function.FunctionFrameworkSupport;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.MetadataParams;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class JythonFunctionFramework implements FunctionFramework, FunctionFrameworkSupport {

  @Getter
  private final String id = "jython";

  @Getter
  private final JavaParameterSet<MetadataParams> parameterSet = JavaParameterSet.fromBindingClass(MetadataParams.class);

  @Override
  public ResultOrProblems<IdentifiedFunction> build(String functionName, Project project, BoundParameters bound) {
    // build the function with the given metadata
    MetadataParams params = parameterSet.bindToObject(bound).getBoundToObject();

    try (JythonFactory factory = new JythonFactory(project)) {
      return loadFunctionFromParams(project.getEngine(), params)
      .flatMap(resource -> {
        return factory.buildFunction(resource, params.toFunctionMetadata(functionName));
      }).composeProblems(Problems.foundWith(IdentifiedFunction.class, functionName));
    }
  }

  @Override
  public boolean canBuild(Project project, Map<String, List<?>> unbound) {
    return MetadataParams.locationEndsWith(unbound, ".py");
  }
}
