/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.cursor;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.hdf5.Hdf5BaseTest;
import nz.org.riskscape.hdf5.types.H5CompoundType;

public class H5FixedSizeCursorTest extends Hdf5BaseTest {

  H5FixedSizeCursor cursor;

  @Test
  public void canIterateOverSimpleType() {
    toCursor("h5_test_data_simple.h5", "data");

    long expected = 0;
    int count = 0;
    while (cursor.hasNext()) {
      assertEquals(expected, cursor.next());
      ++count;
      ++expected;
    }
    assertEquals(950, count);
  }

  @Test
  public void setCurrentIndexAllowsRandomAccess() {
    toCursor("h5_test_data_simple.h5", "data");

    cursor.setCurrentIndex(50);
    assertEquals(50L, cursor.next());

    cursor.setCurrentIndex(920);
    assertEquals(920L, cursor.next());
    assertEquals(921L, cursor.next());
    assertEquals(922L, cursor.next());

    cursor.setCurrentIndex(450);
    assertEquals(450L, cursor.next());

  }

  @Test
  public void canSkipAndPeekCompoundType() {
    toCursor("h5_test_data_compound.h5", "data");

    H5CompoundType cType = (H5CompoundType)dataset.getDataType();

    cursor.setCurrentIndex(50);
    assertEquals(50L, cursor.peek(cType.findMember("x")));
    assertEquals(50D / 1000D, (double)cursor.peek(cType.findMember("y")), 0.0001D);

    cursor.skip();
    cursor.skip();
    assertEquals(52L, cursor.peek(cType.findMember("x")));
    assertEquals(52D / 1000D, (double)cursor.peek(cType.findMember("y")), 0.0001D);

    cursor.setCurrentIndex(99);
    assertEquals(99L, cursor.peek(cType.findMember("x")));
    assertEquals(99D / 1000D, (double)cursor.peek(cType.findMember("y")), 0.0001D);

    cursor.skip();
    cursor.skip();
    assertEquals(101L, cursor.peek(cType.findMember("x")));
    assertEquals(101D / 1000D, (double)cursor.peek(cType.findMember("y")), 0.0001D);

    Object peeked = cursor.peek();
    assertThat(cursor.next(), is(peeked));

  }

  @Test
  public void canProcessATwoDimensionalDataset() {
    // we open the data set with a very small cursor size. It should be ignored anyway for multi dimensioned
    // datasets but good to ensure that doesn't create problems.
    toCursor("h5ex_d_soint.h5", "DS1", 32 * 64 * 4);

    // ensure the dataset is two dimensional
    assertEquals(2, dataset.getDataSpace().numDimensions());

    // spot check some values
    assertEquals(0L, cursor.next());
    assertEquals(-1L, cursor.next());
    assertEquals(-2L, cursor.next());

    cursor.setCurrentIndex(63);
    assertEquals(-63L, cursor.next());
    assertEquals(0L, cursor.next());
    assertEquals(0L, cursor.next());

    cursor.setCurrentIndex(31 * 64);
    assertEquals(0L, cursor.next());
    assertEquals(30L, cursor.next());
    assertEquals(60L, cursor.next());
  }

  @Test
  public void readingMultiDimensionalDataIntoSmallBufferIsAProblem() {
    RiskscapeException ex = assertThrows(RiskscapeException.class, () -> toCursor("h5ex_d_soint.h5", "DS1", 512));
    assertThat(ex.getProblem(), is(
        H5DatasetCursor.PROBLEMS.readSizeTooSmallForMultiDimensionedData(32 * 64 * 4, 512)));
  }

  private void toCursor(String file, String datasetName) {
    toCursor(file, datasetName, null);
  }

  private void toCursor(String file, String datasetName, Integer cursorSizeBytes) {
    this.dataset = openDataset(file, "/" + datasetName);
    if (cursorSizeBytes == null) {
      cursorSizeBytes = (int)(dataset.getDataType().getDataSize() * 100);
    }
    this.cursor = new H5FixedSizeCursor(dataset, dataset.getDataSpace(), dataset.getDataType(), cursorSizeBytes);
  }

}
