/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.types;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

import hdf.hdf5lib.H5;
import hdf.hdf5lib.HDF5Constants;
import lombok.Getter;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

public class H5StringType extends H5Type {

  @Getter
  private boolean variableLength;

  /**
   * currently HDF5 only seems to support the ascii character-set
   */
  private final int characterSet;

  protected H5StringType(long ptr) {
    super(ptr);
    this.variableLength = H5.H5Tis_variable_str(ptr);
    this.characterSet = H5.H5Tget_cset(ptr);
  }

  @Override
  public Type toType() {
    return Types.TEXT;
  }

  @Override
  public Object read(ByteBuffer outBuf, int offset) {
    if (characterSet != HDF5Constants.H5T_CSET_ASCII || variableLength) {
      throw new RuntimeException("Read not yet supported for this string type");
    }
    // turn the raw bytes that we're interested in into an ascii String
    return new String(outBuf.array(), offset, (int) getDataSize(), StandardCharsets.US_ASCII).trim();
  }

  @Override
  public String toString() {
    return "H5T_String";
  }

}
