/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.types;

import java.nio.ByteBuffer;
import java.util.Arrays;

import hdf.hdf5lib.H5;
import nz.org.riskscape.engine.types.Enumeration;
import nz.org.riskscape.engine.types.Type;

public class H5EnumType extends H5Type {

  /**
   * H5 library wants to know the probably length of the name, no idea, let's guess
   */
  private static final int ENUM_NAME_LENGTH = 100;

  private H5IntegerType superType;
  String[] members;

  protected H5EnumType(long id) {
    super(id);
    superType = (H5IntegerType) H5Type.build(H5.H5Tget_super(id));
    int numFields = H5.H5Tget_nmembers(id);
    members = new String[numFields];
    for (int i = 0; i < members.length; i++) {
      members[i] = H5.H5Tenum_nameof(id, superType.intToBytes(i), ENUM_NAME_LENGTH).intern();
    }
  }

  @Override
  public Type toType() {
    return Enumeration.zeroBased(members);
  }

  @Override
  public Object read(ByteBuffer outBuf, int offset) {
    Number index = (Number) superType.read(outBuf, offset);
    return members[index.intValue()];
  }

  @Override
  public String toString() {
    return getClassName() + Arrays.asList(members);
  }

  @Override
  public void close() {
    super.close();
    superType.close();
  }

  @Override
  public boolean isVariableLength() {
    return false;
  }
}
