/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.types;

import static hdf.hdf5lib.H5.*;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;

import hdf.hdf5lib.H5;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.hdf5.H5CompoundMember;

import lombok.Getter;

public class H5CompoundType extends H5Type {
  @Getter
  private final H5CompoundMember[] membersArray;

  protected H5CompoundType(long id) {
    super(id);
    int numFields = H5.H5Tget_nmembers(ptr);
    membersArray = new H5CompoundMember[numFields];
    int byteCount = 0;
    for (int i = 0; i < numFields; i++) {
      String memberName = H5Tget_member_name(ptr, i);
      long memberTypeId = H5Tget_member_type(ptr, i);
      membersArray[i] = new H5CompoundMember(i, memberName, H5Type.build(memberTypeId), this, byteCount);
      byteCount += membersArray[i].type.getDataSize();
    }
  }

  public int findIndex(String string) {
    for (int i = 0; i < getMembersArray().length; i++) {
      if (getMembersArray()[i].name.equals(string)) {
        return i;
      }
    }
    return -1;
  }


  public H5CompoundMember findMember(String string) {
    int index = findIndex(string);
    if (index < 0) {
      return null;
    }
    return getMembersArray()[index];
  }

  public List<H5CompoundMember> getMembers() {
    return Arrays.asList(getMembersArray());
  }

  @Override
  public Object read(ByteBuffer outBuf, int offset) {
    Object[] reading = new Object[getMembersArray().length];

    for (int i = 0; i < reading.length; i++) {
      H5CompoundMember member = getMembersArray()[i];
      reading[i] = member.type.read(outBuf, offset);
      offset += member.type.getDataSize();
    }

    return reading;
  }

  @Override
  public Type toType() {
    Struct.StructBuilder builder = new Struct.StructBuilder();
    for (H5CompoundMember member : getMembersArray()) {
      builder.add(member.name, member.type.toType());
    }
    return builder.build();
  }

  @Override
  public String toString() {
    return getClassName() + " " + getMembers();
  }

  @Override
  public void close() {
    for (H5CompoundMember member : getMembersArray()) {
      member.type.close();
    }
    super.close();
  }

  @Override
  public boolean isVariableLength() {
    for (H5CompoundMember member : getMembersArray()) {
      if (member.type.isVariableLength()) {
        return true;
      }
    }
    return false;
  }

  public int getByteOffsetFor(int memberIndex) {
    int counter = 0;
    for (int i = 0; i < memberIndex; i++) {
      counter += (int) getMembersArray()[i].type.getDataSize();
    }
    return counter;

  }
}
