/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.types;

import static hdf.hdf5lib.H5.*;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import com.google.common.primitives.Longs;

import lombok.Getter;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;

/**
 * Represents an H5 array that could be multi-dimensional.
 *
 * When mapped to a RiskScape type a flattened list is returned. That is one list containing items
 * from dimension1, dimension2 ... etc.
 */
public class H5ArrayType extends H5Type {

  @Getter
  private H5Type superType;

  private final long[] dimensions;

  /**
   * The total number of items in this array across all dimensions.
   */
  private final int itemCount;

  protected H5ArrayType(long id) {
    super(id);
    this.superType = H5Type.build(H5Tget_super(id));

    // compute these up front once to make read efficient - every jni call costs
    int numDims = H5Tget_array_ndims(ptr);
    dimensions = new long[numDims];
    H5Tget_array_dims(ptr, dimensions);
    // we multiply the dimensions to the total items in the array. e.g a 3 x 5 array has 15 items in it
    itemCount = getDimensions().stream().reduce((a, b) -> a * b).orElse(0L).intValue();
  }

  public List<Long> getDimensions() {
    return Longs.asList(dimensions);
  }

  @Override
  public Object read(ByteBuffer outBuf, int offset) {
    // TODO support native arrays, not object pointers?
    List<Object> reading = new ArrayList<>(itemCount);
    for (int i = 0; i < itemCount; i++) {
      reading.add(superType.read(outBuf, offset));
      offset += superType.getDataSize();
    }

    return reading;
  }

  /**
   * Efficient alternative to read for returning whatever numeric value is held in in the super type in to an array of
   * floats.
   * @throws ClassCastException if h5 super type is not numeric
   */
  public float[] readFloats(ByteBuffer outBuf, int offset) {
    float[] reading = new float[itemCount];
    for (int i = 0; i < reading.length; i++) {
      reading[i] = ((Number) superType.read(outBuf, offset)).floatValue();
      offset += superType.getDataSize();
    }

    return reading;
  }

  @Override
  public Type toType() {
    return RSList.create(getSuperType().toType());
  }

  @Override
  public String toString() {
    return getClassName() + "<" + getSuperType().toString() + ">" + getDimensions();
  }

  @Override
  public void close() {
    super.close();
    this.superType.close();
  }

  @Override
  public boolean isVariableLength() {
    // TODO variable length arrays might need detecting here, as well as checking the super type?
    return superType.isVariableLength();
  }

}
