/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.relation;

import java.util.Optional;
import java.util.function.Function;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.BaseRelation;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.hdf5.H5DataSpace;
import nz.org.riskscape.hdf5.H5Dataset;
import nz.org.riskscape.hdf5.cursor.H5DatasetCursor;
import nz.org.riskscape.hdf5.cursor.H5FixedSizeCursor;

/**
 * A relation that emits tuples formed from values in a one dimensional hdf5 dataset.
 */
public class H5Relation extends BaseRelation {

  private final H5Dataset dataset;
  private final Function<H5DatasetCursor, Tuple> tupleMapper;
  private final int cursorSizeMB;

  /**
   * Creates a relation from a HDF5 dataset.
   *
   * @param dataset       the HDF5 dataset (within file) to produce the relation from
   * @param producedType  the type to be produced by this relation
   * @param tupleMapper   a function that accepts a byte buffer and element index and returns a tuple of producedType
   * @param cursorSizeMB  the size in megabytes of each HDF5 read
   */
  public H5Relation(H5Dataset dataset, Struct producedType,
      Function<H5DatasetCursor, Tuple> tupleMapper, int cursorSizeMB) {
    super(producedType);
    this.dataset = dataset;
    this.tupleMapper = tupleMapper;
    this.cursorSizeMB = cursorSizeMB;
  }

  private H5Relation(H5Dataset dataset, Function<H5DatasetCursor, Tuple> tupleMapper,
      int cursorSizeMB, Fields fields) {
    super(fields);
    this.dataset = dataset;
    this.tupleMapper = tupleMapper;
    this.cursorSizeMB = cursorSizeMB;
  }

  @Override
  protected TupleIterator rawIterator() {
    H5DataSpace newDataSpace = dataset.newDataSpace();
    // TODO this fails nastily when we try and open a dataset with vlen elements in it
    H5FixedSizeCursor cursor = new H5FixedSizeCursor(dataset, newDataSpace, dataset.getDataType(),
        cursorSizeMB * 1024 * 1024);
    return new TupleIterator() {
      @Override
      public boolean hasNext() {
        return cursor.hasNext();
      }

      @Override
      public Tuple next() {
        Tuple next = tupleMapper.apply(cursor);
        cursor.skip();
        return next;
      }

      @Override
      public void close() {
        newDataSpace.close();
      }

    };
  }

  @Override
  protected BaseRelation clone(Fields fields) {
    return new H5Relation(dataset, tupleMapper, cursorSizeMB, fields);
  }

  @Override
  public String getSourceInformation() {
    return dataset.toString();
  }

  @Override
  public Optional<Long> size() {
    // each element in the dataset will be read out as a separate tuple
    return Optional.of(dataset.numElements());
  }
}
