/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5.cursor;

import nz.org.riskscape.hdf5.types.H5Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

import java.util.Iterator;

import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.hdf5.H5CompoundMember;
import nz.org.riskscape.hdf5.H5DataSpace;
import nz.org.riskscape.hdf5.H5Dataset;

/**
 * Iterates over elements of an {@link H5Dataset}. Intended to be used primarily as an {@link Iterator}
 * but also allows for random access.
 */
public abstract class H5DatasetCursor implements Iterator<Object> {

  public interface LocalProblems extends ProblemFactory {

    /**
     * When the user defined read size is not big enough to fit all of a multi-dimensioned dataset.
     */
    Problem readSizeTooSmallForMultiDimensionedData(long requiredSize, long availableSize);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  protected final long numElements;
  protected final H5Dataset dataset;
  protected final H5DataSpace dataspace;
  protected final H5Type type;


  public H5DatasetCursor(H5Dataset dataset, H5DataSpace dataspace, H5Type type) {
    this.dataset = dataset;
    this.dataspace = dataspace;
    this.type = type;
    this.numElements = dataspace.numElements();
  }

  public long size() {
    return numElements;
  }

  /**
   * @return same as size(), but throws an exception if the dataspace size
   * doesn't fit in an integer value
   */
  public int sizeAsInt() {
    return dataspace.numElementsAsInt();
  }

  /**
   * Gets the current position of the cursor within the dataset elements.
   * @return the index of the element that will be returned by next()
   */
  public abstract long getCurrentIndex();

  /**
   * Resets the position of the cursor to the given element index.
   * @param index the index of the element that should be returned by next()
   */
  public abstract void setCurrentIndex(long index);

  /**
   * Advances the cursor index one step. Similar to calling {@link #next() } but without unmarshalling any data
   */
  public abstract void skip();

  /**
   * Same as {@link #next() } but the cursor position is not changed.
   * @return object read from the current cursor position
   */
  public abstract Object peek();

  /**
   * Read the data for member only. This can be more efficient than reading the data with {@link #next() }.
   * Peeking does not affect the cursor position.
   *
   * Note that it is up to the caller to ensure that member is in fact a member of this cursor's data type.
   * If it it not then unexpected results are likely.
   *
   * @param member member to read
   * @return the value of member at the current cursor position
   */
  public abstract Object peek(H5CompoundMember member);
}
