/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5;

import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import hdf.hdf5lib.H5;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.plugin.Plugin;
import nz.org.riskscape.engine.plugin.PluginDescriptor;
import nz.org.riskscape.engine.plugin.PluginFeature;
import nz.org.riskscape.engine.plugin.PluginRuntimeException;
import nz.org.riskscape.engine.spi.CliCommand;
import nz.org.riskscape.problem.Problem;

@Slf4j
public class Hdf5Plugin extends Plugin {

  public static final Parameter SETTING_LIB_PATH = Parameter.required("hdf5.lib_path", String.class);

  @Getter
  private final List<PluginFeature> features = Arrays.asList(
    new CliCommand(Hdf5Command.class)
  );

  public Hdf5Plugin(PluginDescriptor pDescriptor) {
    super(pDescriptor);
  }

  @Override
  public void startUp(Map<String, List<String>> settings) throws PluginRuntimeException {
    loadLibrary(settings);
  }

  private void loadLibrary(Map<String, List<String>> settings) {
    List<String> optionalPath = settings.get(SETTING_LIB_PATH.getName());
    if (optionalPath != null && !optionalPath.isEmpty()) {
      String path = Paths.get(optionalPath.get(0)).toAbsolutePath().toString();
      System.setProperty(H5.H5PATH_PROPERTY_KEY, path);
      log.info("{} is set to {}, setting H5 path for H5.loadH5Lib...", SETTING_LIB_PATH.getName(), path);
    } else {
      log.info("{} is not set, relying on system load path to find hdf5...", SETTING_LIB_PATH.getName());
    }

    try {
      H5.loadH5Lib();
      H5.H5open();
    } catch (Throwable e) {
      throw new PluginRuntimeException(e, "Failed to load hdf5 library - %s", e.getMessage());
    }

  }

  @Override
  public List<Problem> initializeEngine(Engine engine) {
    engine.getBookmarkResolvers().add(new Hdf5Resolver(engine));
    return Collections.emptyList();
  }
}
