/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.hdf5;

import java.nio.file.Path;
import java.util.function.Consumer;
import java.util.function.Function;

import hdf.hdf5lib.H5;
import hdf.hdf5lib.HDF5Constants;

/**
 * Gives access to datasets stored within an hdf5 file on the filesystem.
 */
public class H5File extends H5Object {

  public H5File(Path path) {
    this(path.toString());
  }

  public H5File(String filename) {
    super(H5.H5Fopen(filename, HDF5Constants.H5F_ACC_RDONLY, HDF5Constants.H5P_DEFAULT));
  }

  /**
   * @return the root {@link H5Group.  You are responsible for closing this group, maybe use acceptRootGroup or
   * applyRootGroup for callback versions that close the group for you
   */
  public H5Group getRootGroup() {
    return H5Group.getRootGroup(H5.H5Gopen(ptr, "/", HDF5Constants.H5P_DEFAULT));
  }

  /**
   * Convenience version of getRootGroup that handles closing the root group object for you.
   * Applies a function to the root group, returning the result after closing the root group object
   */
  public <T> T applyRootGroup(Function<H5Group, T> callback) {
    try (H5Group root = getRootGroup()) {
      return callback.apply(root);
    }
  }

  /**
   * Convenience version of getRootGroup that handles closing the root group object for you.
   * Applies a callback to the root group, closing the root group object once it has run
   */
  public void acceptRootGroup(Consumer<H5Group> callback) {
    try (H5Group root = getRootGroup()) {
      callback.accept(root);
    }
  }

  /**
   * @return either an {@link H5Group} or {@link H5Dataset} by its absolute path in this h5file
   */
  public H5Object openObject(String path) {

    if (path.equals("/")) {
      return getRootGroup();
    }

    // drop the leading slash - it just makes it absolute to the root group
    if (path.startsWith("/")) {
      path = path.substring(1);
    }

    return getRootGroup().openChild(path);
  }

  public H5Dataset openDataset(H5DatasetPath datasetId) {
    return openDataset(datasetId.groupName, datasetId.datasetName);
  }

  public H5Dataset openDataset(String groupPath, String datasetName) {
    long groupId = H5.H5Gopen(ptr, groupPath, HDF5Constants.H5P_DEFAULT);
    try {
      long datasetId = H5.H5Dopen(groupId, datasetName, HDF5Constants.H5P_DEFAULT);
      return new H5Dataset(datasetId, datasetName);
    } finally {
      H5.H5Gclose(groupId);
    }
  }


  @Override
  public void close() {
    H5.H5Fclose(ptr);
  }
}
