/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;
import org.locationtech.jts.geom.GeometryFactory;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class CsvRelationTest extends ProjectTest implements CrsHelper {

  CoordinateReferenceSystem crs = nzMapGrid();
  GeometryFactory geometryFactory = project.getSridSet().getGeometryFactory(crs);

  @Test
  public void canConvertABasicCsvToTuples() {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "foo1,bar1,baz1%n"
        + "foo2,bar2,baz2%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    Struct expectedType = Struct.of("foo", Types.TEXT).and("bar", Types.TEXT).and("baz", Types.TEXT);
    assertEquals(expectedType, csvRelation.getType());

    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(expectedType, "foo1", "bar1", "baz1"),
        Tuple.ofValues(expectedType, "foo2", "bar2", "baz2")
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
    assertThat(csvRelation.size().get(), is(2L));
  }

  @Test
  public void canAddLineNumbersToTuples() {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "%n"
        + "%n"
        + "foo1,bar1,baz1%n"
        + "%n"
        + "foo2,bar2,baz2%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes(), true);
    Struct expectedType =
        Struct.of("foo", Types.TEXT, "bar", Types.TEXT, "baz", Types.TEXT, "__linenum", Types.INTEGER);

    assertEquals(expectedType, csvRelation.getType());

    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(expectedType, "foo1", "bar1", "baz1", 4L),
        Tuple.ofValues(expectedType, "foo2", "bar2", "baz2", 6L)
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
    assertThat(csvRelation.size().get(), is(2L));
  }

  @Test
  public void makeSureMissingColumnsThrowInvalidTuple() {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "foo1,bar1%n"
        + "foo2,bar2,baz2%n");

    CsvRelation csvRelation = (CsvRelation) CsvRelation.fromBytes(csv.getBytes()).skipInvalid(engine.getProblemSink());
    Struct expectedType = Struct.of("foo", Types.TEXT).and("bar", Types.TEXT).and("baz", Types.TEXT);
    assertEquals(expectedType, csvRelation.getType());

    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(expectedType, "foo2", "bar2", "baz2")
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
    // note that size here includes any invalid tuples
    assertThat(csvRelation.size().get(), is(2L));
  }

  @Test
  public void makeSureEmptyStringsAreJustEmptyStringsAndNotNull() {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "foo1,,baz1%n"
        + "foo2,bar2,baz2%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    Struct type = csvRelation.getType();

    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(type, "foo1", "", "baz1"),
        Tuple.ofValues(type, "foo2", "bar2", "baz2")
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
  }

  @Test
  public void makeSureTheStringsNullIsJustAString() {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "foo1,null,baz1%n"
        + "foo2,bar2,NULL%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    Struct type = csvRelation.getType();

    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(type, "foo1", "null", "baz1"),
        Tuple.ofValues(type, "foo2", "bar2", "NULL")
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
  }


  @Test
  public void canDoSomeFruitSaladMappingAndRestricting() throws Exception {
    String csv = String.format(""
        + "foo,bar,baz%n"
        + "foo1,,baz1%n"
        + "foo2,bar2,baz2%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    Struct originalType = csvRelation.getType();



    List<Tuple> expectedTuples = Arrays.asList(
        Tuple.ofValues(originalType, "foo1", "", "baz1"),
        Tuple.ofValues(originalType, "foo2", "bar2", "baz2")
    );
    List<Tuple> collected = csvRelation.stream().collect(Collectors.toList());
    assertEquals(expectedTuples, collected);
    assertFalse(csvRelation.getSpatialMetadata().isPresent());
  }

  @Test
  public void correctlySkipsEmptyRows() throws Exception {
    String csv = String.format("%n"
        + "foo,bar,baz%n"
        + "%n"
        + "foo1,,baz1%n"
        + "foo2,bar2,baz2%n%n");

    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    assertEquals(2, csvRelation.stream().collect(Collectors.toList()).size());
    assertThat(csvRelation.size().get(), is(2L));
  }

  @Test
  public void canReadEmptyCsvFile() throws Exception {
    String csv = String.format("%n"
        + "foo,bar,baz%n"
        + "%n");
    CsvRelation csvRelation = CsvRelation.fromBytes(csv.getBytes());
    assertEquals(0, csvRelation.stream().collect(Collectors.toList()).size());
    assertThat(csvRelation.size().get(), is(0L));
  }
}
