/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import org.junit.Test;

import nz.org.riskscape.defaults.DefaultProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.TupleMatchers;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.sched.SchedulerBasedExecutor;
import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;

/**
 * Another set of pipeline integration tests that makes use of bits that are only available in the default plugin
 * @author nickg
 *
 */
public class PipelineIntegrationTest extends DefaultProjectTest {

  @Test
  public void testAPipelineWithLotsOfBranchesAndGroupingCombinations() throws Exception {
    ListRelation listRelation = ListRelation.ofValues(
      LongStream.rangeClosed(1, 1000).mapToObj(Long::valueOf).collect(Collectors.toList()).toArray()
    );
    addPickledData("integers", listRelation);
    PipelineDeclaration pipelineAst = parsePipelineFile("src/test/resources/pipelines/various-groupings.txt");
    RealizedPipeline realized = executionContext.realize(pipelineAst);

    SchedulerBasedExecutor executor = new SchedulerBasedExecutor(engine);
    executor.setNumThreads(4);
    TestPipelineJobContext context = new TestPipelineJobContext(realized);
    executor.execute(context).get();

    List<Tuple> countsAndSumsOfPairedSink = context.getStepToSink().get("counts_and_sums_of_paired").getTuples();
    assertThat(
      countsAndSumsOfPairedSink,
      contains(
        allOf(
          TupleMatchers.tupleWithValue("count", equalTo(1000L)),
          TupleMatchers.tupleWithValue("sum", equalTo(0L))
        )
      )
    );

    List<Tuple> sumOfModuloThrees = context.getStepToSink().get("sum_of_modulo_threes").getTuples();
    assertThat(
      sumOfModuloThrees,
      containsInAnyOrder(
        TupleMatchers.tupleWithValue("count", equalTo(400L)),
        TupleMatchers.tupleWithValue("count", equalTo(300L)),
        TupleMatchers.tupleWithValue("count", equalTo(300L))
      )
    );

    List<Tuple> groupedInThrees = context.getStepToSink().get("grouped_in_threes").getTuples();
    assertThat(
      groupedInThrees,
      containsInAnyOrder(
        allOf(
          TupleMatchers.tupleWithValue("modulo", equalTo(0D)),
          TupleMatchers.tupleWithValue("integers", hasItems(3L, 6L, 999L))

        ),
        allOf(
          TupleMatchers.tupleWithValue("modulo", equalTo(1D)),
          TupleMatchers.tupleWithValue("integers", hasItems(1L, 4L, 1000L))
        ),
        allOf(
          TupleMatchers.tupleWithValue("modulo", equalTo(2D)),
          TupleMatchers.tupleWithValue("integers", allOf(hasItems(2L, 5L, 998L)))
        )
      )
    );
  }

  private PipelineDeclaration parsePipelineFile(String path) throws IOException {
    return PipelineParser.INSTANCE
        .parsePipeline(Files.readAllLines(Paths.get(path)).stream().collect(Collectors.joining("\n")));
  }
}
