/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.geometry;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.junit.Test;
import org.locationtech.jts.geom.Geometry;
import org.mockito.Mockito;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.Pair;

public class SampleCoverageTest extends BaseSampleCoverageTest {

  public SampleCoverageTest() {
    function = new SampleCoverage();
  }

  @Test
  public void willReturnTheResultOfTheCoveragesIntersectionOperation() throws Exception {
    realize();
    Type returnType = realized.getReturnType();
    Struct structType = returnType.find(RSList.class).get().getMemberType().find(Struct.class).get();
    assertEquals(RSList.create(Struct.of("geometry", Types.GEOMETRY, "sampled", coverageMemberType)), returnType);


    Function<Geometry, List<Pair<Geometry, Object>>> op = (geom) -> Arrays.asList(
        Pair.of(geom, "Hi mom"),
        Pair.of(point2, "How is dad?")
    );

    Mockito.when(coverage.getEvaluateIntersectionOp()).thenReturn(Optional.of(op));

    assertEquals(
        Arrays.asList(
            Tuple.ofValues(structType, point1, "Hi mom"),
            Tuple.ofValues(structType, point2, "How is dad?")
        ),
        realized.call(Arrays.asList(point1, coverage))
    );
  }


  @Test
  public void willThrowAnExceptionIfTheCoverageDoesNotSupportIntersectionSampling() {
    Mockito.when(coverage.getEvaluateIntersectionOp()).thenReturn(Optional.empty());
    realize();

    RiskscapeException ex = Assert.assertThrows(RiskscapeException.class,
        () -> realized.call(Arrays.asList(point1, coverage)));
    assertThat(ex.getProblem(), is(GeneralProblems.get().operationNotSupported("intersection", coverage.getClass())));
  }

}
