/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.junit.Test;

import javax.xml.parsers.DocumentBuilder;

import org.eclipse.emf.ecore.EObject;
import org.geotools.data.wfs.internal.WFSGetCapabilities;
import org.geotools.xsd.Configuration;
import org.geotools.xsd.Parser;

import nz.org.riskscape.engine.ProjectTest;

public class WfsBookmarkResolverTest extends ProjectTest {

  WfsBookmarkResolver resolver;
  Parser parser;
  DocumentBuilder documentBuilder;

  WFSGetCapabilities capsGetFeature100;
  WFSGetCapabilities capsOperationsMetadata1000;
  WFSGetCapabilities capsNoCountDefault;

  public WfsBookmarkResolverTest() throws Exception {
    resolver = new WfsBookmarkResolver(engine);

    Configuration configuration = new org.geotools.wfs.v2_0.WFSCapabilitiesConfiguration();
    parser = new Parser(configuration);

    // populate the test capabilities. note that these test xml files have been stripped out to remove
    // all but <ows:OperationsMetadata> to keep them concise.
    capsGetFeature100 = mockCapabilites("getCapabilies_getFeature_countDefault100.xml");
    capsOperationsMetadata1000
        = mockCapabilites("getCapabilies_operationsMetaData_countDefault1000.xml");
    capsNoCountDefault = mockCapabilites("getCapabilies_noCountDefault.xml");

  }

  @Test
  public void getPageSizeUsesGetFeaturePageSizeIfAvailable() {
    assertThat(resolver.getPageSize(capsGetFeature100, 10000), is(100));
  }

  @Test
  public void getPageSizeUsesOperationsMetadataIfNoGetFeatureCount() {
    assertThat(resolver.getPageSize(capsOperationsMetadata1000, 10000), is(1000));
  }

  @Test
  public void getPageSizeUsesUserSettingWhenNoServerCount() {
    assertThat(resolver.getPageSize(capsNoCountDefault, 10000), is(10000));
  }

  @Test
  public void getPageSizeUsesUserPageSizeIfSmaller() {
    assertThat(resolver.getPageSize(capsGetFeature100, 10), is(10));
  }

  private WFSGetCapabilities mockCapabilites(String from) {
    WFSGetCapabilities capabilities = mock(WFSGetCapabilities.class);

    try {
      when(capabilities.getParsedCapabilities())
          .thenReturn((EObject)parser.parse(
              WfsBookmarkResolverTest.class.getResourceAsStream("/nz/org/riskscape/engine/defaults/data/" + from)
          ));
    } catch (Exception e) {
      throw new RuntimeException(e);
    }

    return capabilities;
  }

}
