/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;

import static org.hamcrest.MatcherAssert.*;
import static org.hamcrest.Matchers.*;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.Map;

import org.geotools.jdbc.JDBCDataStore;
import org.junit.Before;
import org.junit.Test;

public class RsGpkgDataStoreFactoryTest {

  // I got this file from a failing project in the platform and checked it in as read only to avoid some other process
  // from cleaning up the wal-mode state that causes this test to fail
  Path databasePath = Paths.get("src/test/resources/geopackage/wal-mode-1291.gpkg");
  Path tempDirectory;
  Path target;
  RsGpkgDataStoreFactory factory = new RsGpkgDataStoreFactory();

  @Before
  public void setup() throws IOException {
    tempDirectory = Files.createTempDirectory("rs-test");
    target = tempDirectory.resolve("fail.gpkg");
    Files.copy(databasePath, target);
  }

  @Test
  public void canReadAWalModeFileOnAReadOnlyFilesystem() throws Exception {
    // change the file and the directory to be read only - this mimics the appearance of a read only filesystem
    Files.setPosixFilePermissions(target,
        EnumSet.of(PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ, PosixFilePermission.OWNER_READ)
    );
    Files.setPosixFilePermissions(tempDirectory,
      EnumSet.of(PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ, PosixFilePermission.OWNER_READ,
          PosixFilePermission.OWNER_EXECUTE)
    );

    JDBCDataStore dataStore = factory.createDataStore(Map.of("database", target, "read_only", true));
    // check we can read it
    assertThat(Arrays.asList(dataStore.getTypeNames()), contains("gadm40_vut_1_province"));

    // check that we did in fact access it with the immutable flag
    String url = dataStore.getDataSource().getConnection().getMetaData().getURL();
    assertThat(url, endsWith("?immutable=true"));
  }

  @Test
  public void canReadAWalModeFileOnAWriteableFilesystem() throws Exception {
    JDBCDataStore dataStore = factory.createDataStore(Map.of("database", target, "read_only", true));

    // check we can read it
    assertThat(Arrays.asList(dataStore.getTypeNames()), contains("gadm40_vut_1_province"));

    // but no immutable flag this time
    String url = dataStore.getDataSource().getConnection().getMetaData().getURL();
    assertThat(url, not(endsWith("?immutable=true")));

  }

}
