/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;


import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.geotools.referencing.CRS;
import org.junit.Test;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;

import nz.org.riskscape.engine.GeometryMatchers;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.StructMatchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.TupleMatchers;
import nz.org.riskscape.engine.TypeMatchers;
import nz.org.riskscape.engine.data.BookmarkProblems;
import nz.org.riskscape.engine.gt.BaseGeometryHelper;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.MultiGeom;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;


public class GeoPackageRelationResolverTest extends BaseBookmarkResolverTest {

  public GeoPackageRelationResolverTest() {
    super("geopackage", engine -> new GeoPackageRelationResolver(engine));
  }

  @Test
  public void canReadPoints() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "points")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom", TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, Types.POINT)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        allOf(
          TupleMatchers.tupleWithValue("name", is("point")),
          TupleMatchers.tupleWithValue("geom", GeometryMatchers.isInCrs(project.getSridSet(), SRIDSet.EPSG4326_LONLAT)),
          TupleMatchers.tupleWithValue("geom", GeometryMatchers.isGeometry(crs84Helper.fromWkt("POINT (10 10)")))
        )
    ));
    assertThat(relation.size().get(), is(1L));
  }

  @Test
  public void canReadPointInNZTM() throws Exception {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "nz-points")).get();
    Struct relationType = relation.getType();

    // Geopackages store geometries with an XY axis (same as shapefiles)
    CoordinateReferenceSystem nztmXy = CRS.decode("EPSG:2193", true);
    BaseGeometryHelper nztmXyHelper = new BaseGeometryHelper(project.getSridSet(), nztmXy);

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom", TypeMatchers.isReferencedType(nztmXy, Types.POINT)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        allOf(
          TupleMatchers.tupleWithValue("name", is("auckland")),
          TupleMatchers.tupleWithValue("geom", GeometryMatchers.isInCrs(project.getSridSet(), nztmXy)),
          TupleMatchers.tupleWithValue("geom",
              GeometryMatchers.isGeometry(nztmXyHelper.fromWkt("POINT (1690000 6015000)")))
        ),
        allOf(
          TupleMatchers.tupleWithValue("name", is("wellington")),
          TupleMatchers.tupleWithValue("geom", GeometryMatchers.isInCrs(project.getSridSet(), nztmXy)),
          TupleMatchers.tupleWithValue("geom",
              GeometryMatchers.isGeometry(nztmXyHelper.fromWkt("POINT (1684000 5460000)")))
        )
    ));
    assertThat(relation.size().get(), is(2L));
  }

  @Test
  public void canReadLines() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "lines")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom", TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, Types.LINE)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.fromWkt("LINESTRING (10 10,20 10)"), "line")
    ));
  }

  @Test
  public void canReadPolygons() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "polygons")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom", TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, Types.POLYGON)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.fromWkt("POLYGON ((10 10,10 20,20 20,20 10,10 10))"), "polygon"),
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("POLYGON ((50 50,50 60,60 60,60 50,50 50),(55 55,56 55,56 56,55 56,55 55))"),
            "keyed-polygon")
    ));
  }

  @Test
  public void canReadMultiPoints() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "mpoints")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom",
            TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, MultiGeom.MULTI_POINT)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.fromWkt("MULTIPOINT ((10 10),(20 20))"), "mpoint")
    ));
  }

  @Test
  public void canReadMultiLines() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "mlines")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom",
            TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, MultiGeom.MULTI_LINE)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.fromWkt("MULTILINESTRING ((10 10,20 10),(20 20,30 20))"), "mline")
    ));
  }

  @Test
  public void canReadMultiPolygons() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "mpolygons")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom",
            TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, MultiGeom.MULTI_POLYGON)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, containsInAnyOrder(
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("MULTIPOLYGON (((10 10,10 20,20 20,20 10,10 10)),((50 50,50 60,60 60,60 50,50 50)))"),
            "mpolygon"),
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("MULTIPOLYGON (((10 10,10 20,20 20,20 10,10 10)),"
                + "((50 50,50 60,60 60,60 50,50 50),(55 55,56 55,56 56,55 56,55 55)))"),
            "keyed-mpolygon")
    ));
  }

  @Test
  public void canReadMixedGeometryTypes() {
    Relation relation = from("test.gpkg", ImmutableMap.of("layer", "mixed")).get();
    Struct relationType = relation.getType();

    assertThat(relationType, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom",
            TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, Types.GEOMETRY)),
        StructMatchers.isStructMember("name", Types.TEXT)
    )));

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, containsInAnyOrder(
        Tuple.ofValues(relationType, crs84Helper.fromWkt("POINT (10 10)"), "point"),
        Tuple.ofValues(relationType, crs84Helper.fromWkt("LINESTRING (10 10,20 10)"), "line"),
        Tuple.ofValues(relationType, crs84Helper.fromWkt("POLYGON ((10 10,10 20,20 20,20 10,10 10))"), "polygon"),
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("POLYGON ((50 50,50 60,60 60,60 50,50 50),(55 55,56 55,56 56,55 56,55 55))"),
            "keyed-polygon"),
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("MULTIPOLYGON (((10 10,10 20,20 20,20 10,10 10)),((50 50,50 60,60 60,60 50,50 50)))"),
            "mpolygon"),
        Tuple.ofValues(relationType,
            crs84Helper.fromWkt("MULTIPOLYGON (((10 10,10 20,20 20,20 10,10 10)),"
                + "((50 50,50 60,60 60,60 50,50 50),(55 55,56 55,56 56,55 56,55 55)))"),
            "keyed-mpolygon")
    ));
  }

  @Test
  public void returnsFirstLayerIfNotSpecified() {
    ResultOrProblems<Relation> relationOr = from("cities.gpkg");
    assertThat(render(relationOr.getProblems()), containsString("no layer parameter given"));
    Relation relation = relationOr.getWithProblemsIgnored();
    Struct relationType = relation.getType();

    assertExpectedCitiesType(relationType);

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.point(151, -33), "Sydney", "Largest city in Australia", 5300000.0D),
        Tuple.ofValues(relationType, crs84Helper.point(174, -36), "Auckland", "Largest city in New Zealand", 1700000D),
        Tuple.ofValues(relationType, crs84Helper.point(174, -41), "Wellington", "New Zealand's capital city", 436000D)
    ));
  }

  @Test
  public void returnsRequestedLayer() {
    Relation relation = from("cities.gpkg", ImmutableMap.of("layer", "nz-cities")).get();
    Struct relationType = relation.getType();

    assertExpectedCitiesType(relationType);

    List<Tuple> tuples = relation.iterator().collect(Collectors.toList());
    assertThat(tuples, contains(
        Tuple.ofValues(relationType, crs84Helper.point(174, -36), "Auckland", "Largest city in New Zealand", 1700000D),
        Tuple.ofValues(relationType, crs84Helper.point(174, -41), "Wellington", "New Zealand's capital city", 436000D)
    ));
  }

  @Test
  public void errorIfGeoPackageHasNoFeatureLayers() {
    // a geopackage can have feature layers and/or raster layers.
    assertThat(from("test-raster1_noOgrContents.gpkg", Collections.emptyMap()), failedResult(hasAncestorProblem(
        is(BookmarkProblems.get().noFeatureLayers())
    )));

    // the no feature layers problem takes priority over the unknown layer problem.
    assertThat(from("test-raster1_noOgrContents.gpkg", ImmutableMap.of("layer", "nz-cities")),
        failedResult(hasAncestorProblem(
            is(BookmarkProblems.get().noFeatureLayers())
        )));
  }

  private void assertExpectedCitiesType(Struct got) {
    assertThat(got, StructMatchers.isStruct(Lists.newArrayList(
        StructMatchers.isStructMember("geom", TypeMatchers.isReferencedType(SRIDSet.EPSG4326_LONLAT, Types.POINT)),
        StructMatchers.isStructMember("name", Types.TEXT),
        StructMatchers.isStructMember("desc", Types.TEXT),
        StructMatchers.isStructMember("population", Types.FLOATING)
    )));
  }

}
