/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.util.Arrays;
import java.util.Map;

import org.json.simple.parser.ParseException;
import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.RelationMatchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.resource.ResourceProblems;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.StandardCodes;

public class GeoJSONResolverTest extends BaseBookmarkResolverTest {

  public GeoJSONResolverTest() {
    super("geojson", engine -> new GeoJSONResolver(engine));
  }

  @Test
  public void canReadCollectionOfPoints() {
    Relation relation = from("test.geojson").get();
    Struct expectedType = Struct.of(
        "geometry", Referenced.of(Types.GEOMETRY, crs84Helper.getCrs()),
        "t", Types.TEXT,
        "i", Types.INTEGER,
        "f", Types.FLOATING);

    assertThat(relation.getType(), is(expectedType));

    assertThat(relation, RelationMatchers.withTuples(Arrays.asList(
        Tuple.ofValues(expectedType, crs84Helper.point(15, 10), "foo", 20L, 1.0D),
        Tuple.ofValues(expectedType, crs84Helper.point(25, 20), "bar", 25L, 2.5D)
    )));
    assertThat(relation.size().get(), is(2L));
  }

  @Test
  public void canReadMixOfFloatAndLong() {
    // when both double and long values are present, floating type should win.
    // Some GIS systems return this sort of data a lot (I'm looking at you ArcGIS Online)
    Relation relation = from("mixed_types.geojson").get();
    Struct expectedType = Struct.of(
        "geometry", Referenced.of(Types.GEOMETRY, crs84Helper.getCrs()),
        "f", Types.FLOATING,
        "f2", Types.FLOATING);

    assertThat(relation.getType(), is(expectedType));

    assertThat(relation, RelationMatchers.withTuples(Arrays.asList(
        Tuple.ofValues(expectedType, crs84Helper.point(15, 10), 20D, 1.5D),
        Tuple.ofValues(expectedType, crs84Helper.point(25, 20), 25.5D, 2.0D)
    )));
  }

  @Test
  public void canReadDataWithNulls() {
    Relation relation = from("nulls.geojson").get();
    Struct expectedType = Struct.of(
        "geometry", Referenced.of(Types.GEOMETRY, crs84Helper.getCrs()),
        "t", Types.TEXT,
        "i", Types.INTEGER,
        "f", Types.FLOATING);

    assertThat(relation.getType(), is(expectedType));

    assertThat(relation, RelationMatchers.withTuples(Arrays.asList(
        Tuple.ofValues(expectedType, crs84Helper.point(15, 10), null, null, null),
        Tuple.ofValues(expectedType, crs84Helper.point(25, 20), "bar", 25L, 2.5D)
    )));
  }

  @Test
  public void errorIfNoGeometry() {
    assertThat(from("test.json"), Matchers.failedResult(
        Matchers.hasAncestorProblem(is(Problem.error(StandardCodes.GEOMETRY_REQUIRED,
            Struct.of("t", Types.TEXT, "i", Types.INTEGER, "f", Types.FLOATING))))
    ));
  }

  @Test
  public void errorIfBadSyntax() {
    assertThat(from("invalid-syntax.geojson"), Matchers.failedResult(
        Matchers.hasAncestorProblem(Matchers.isProblem(ParseException.class))
    ));
  }

  @Test
  public void giveGoodErrorOnMissingRemoteFile() throws Exception {
    URI target = URI.create("https://" + HTTPS_TEST_URI + "/bogus/test.geojson");
    Bookmark bookmark = new Bookmark("test", "my test bookmark", "geojson",
        target, Map.of());

    assertThat(
        resolver.resolve(bookmark, bindingContext).get().getData(Relation.class),
        nz.org.riskscape.engine.Matchers.failedResult(
            nz.org.riskscape.engine.Matchers.hasAncestorProblem(is(ResourceProblems.get().notFound(target)))
        )
    );
  }
}
