/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Base64;

import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.resource.Resource.Options;

public class DataURILoaderTest extends ProjectTest {

  DataURILoader loader = new DataURILoader(engine);

  @Test
  public void canLoadSomeSimpleStringData() throws UnsupportedEncodingException {
    String payload = "Once upon a time, there was a handsome prince named Susan. He ruled over the lands left to him " +
        "from his mother, and all in the kingdom were happy.\n\nThe end.";

    URI urlEscaped = URI.create("data:," + URLEncoder.encode(payload, "US-ASCII"));

    assertTrue(loader.canLoad(urlEscaped));
    assertEquals(loader.load(urlEscaped).getContentAsString(), payload);
  }

  @Test
  public void canLoadSomeStringDataEncodedUsingUTF8() throws Exception {
    String payload = "Once upon a time, there was a handsome prince named Tāne. He ruled over the lands left to him " +
        "from his mother, and all in the kingdom were happy.\n\nThe end.";

    URI urlEscaped = URI.create("data:text/plain;charset=utf-8," + URLEncoder.encode(payload, "UTF-8"));

    assertTrue(loader.canLoad(urlEscaped));
    assertEquals(loader.load(urlEscaped).getContentAsString(), payload);
  }

  @Test
  public void canLoadSomeStringDataEncodedWithBase64() throws Exception {
    String payload = "Once upon a time, there was a handsome prince named Graham. He ruled over the lands left to "
        + "him from his mother, and all in the kingdom were happy.\n\nThe end.";

    URI urlEscaped = URI.create("data:;base64," + Base64.getEncoder().encodeToString(payload.getBytes("US-ASCII")));

    assertTrue(loader.canLoad(urlEscaped));
    assertEquals(loader.load(urlEscaped).getContentAsString(), payload);
  }

  @Test
  public void canLoadSomeStringDataEncodedWithBase64AndUTF8() throws Exception {
    String payload = "Once upon a time, there was a handsome prince named Nĩna. He ruled over the lands left to him " +
        "from his mother, and all in the kingdom were happy.\n\nThe end.";

    URI urlEscaped = URI.create("data:text/plain;charset=utf-8;base64," +
        Base64.getEncoder().encodeToString(payload.getBytes("utf-8")));

    assertTrue(loader.canLoad(urlEscaped));
    assertEquals(loader.load(urlEscaped).getContentAsString(), payload);
  }

  @Test
  public void canLoadSomeBinaryDataEncodedWithBase64() throws Exception {
    byte[] payload = new byte[] {1, 2, 3, 4, 5, 100, 101, 102, 127};

    URI urlEscaped = URI.create("data:application/octet-stream;base64," +
        Base64.getUrlEncoder().encodeToString(payload));

    assertTrue(loader.canLoad(urlEscaped));
    assertThat(loader.load(urlEscaped).getContentStream().readAllBytes(), equalTo(payload));
  }

  @Test
  public void canLocalizeSomeBinaryDataEncodedWithBase64() throws Exception {
    byte[] payload = new byte[] {1, 2, 3, 4, 5, 100, 101, 102, 127};

    URI urlEscaped = URI.create("data:application/octet-stream;base64," +
        Base64.getUrlEncoder().encodeToString(payload));

    assertTrue(loader.canLoad(urlEscaped));
    Options options = new Options();
    Path localizied = loader.load(urlEscaped).ensureLocal(options).orElseThrow(() -> new RuntimeException("boom"));
    assertThat(Files.readAllBytes(localizied), equalTo(payload));
  }

}
