/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.defaults.data;

import static org.junit.Assert.*;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import com.google.common.collect.Maps;

import lombok.RequiredArgsConstructor;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.BookmarkResolver;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.engine.defaults.resource.HttpResourceLoader;
import nz.org.riskscape.engine.gt.BaseGeometryHelper;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.GeometryFunctions;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.problem.ResultOrProblems;

@RequiredArgsConstructor
public abstract class BaseBookmarkResolverTest extends ProjectTest {

  protected final String format;
  protected final BookmarkResolver resolver;

  protected final BindingContext context = super.bindingContext;

  protected final BaseGeometryHelper crs84Helper =
      new BaseGeometryHelper(project.getSridSet(), SRIDSet.EPSG4326_LONLAT);

  protected BaseBookmarkResolverTest(String format, Function<Engine, BookmarkResolver> resolverBuilder) {
    this.format = format;
    this.resolver = resolverBuilder.apply(engine);

    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(new GeometryFunctions(engine).getFunctions());
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);

    engine.getResourceFactory().add(new HttpResourceLoader(engine));
  }

  protected ResultOrProblems<Relation> from(String testFile) {
    return from(bookmark(testUri(testFile)));
  }

  protected ResultOrProblems<Relation> from(String testFile, Map<String, String> options) {
    return from(bookmark(testUri(testFile), options));
  }

  protected ResultOrProblems<Relation> from(Bookmark bookmark) {
    try {
    Optional<ResolvedBookmark> resolved = resolver.resolve(bookmark, context);
    assertTrue(resolved.isPresent());

    return resolved.get().getData(Relation.class);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  protected Bookmark bookmark(URI absoluteUrl) {
    return bookmark(absoluteUrl, Collections.emptyMap());
  }

  protected Bookmark bookmark(URI absoluteUrl, Map<String, String> options) {
    return new Bookmark("test", "my test bookmark", format, absoluteUrl,
        Maps.transformValues(options, Collections::singletonList));
  }

  protected URI testUri(String string) {
    try {
      return getClass().getResource("/nz/org/riskscape/engine/defaults/data/" + string).toURI();
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

}
