/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.geo;

import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Test;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;

import nz.org.riskscape.engine.CrsHelper;

public class SegmentLineStringOpTest implements CrsHelper {

  GeometryFactory gf = new GeometryFactory();
  SegmentLineStringOp op = new SegmentLineStringOp();

  LineString singleEdgeLineString = gf.createLineString(new Coordinate[] {
      new Coordinate(-1, 0),
      new Coordinate(1, 0) // 2
  });

  LineString doubleEdgeLineString = gf.createLineString(new Coordinate[] {
      new Coordinate(-1, 0),
      new Coordinate(1, 0), // 2
      new Coordinate(1, 1) // 1
  });

  LineString manyEdgeLineString = gf.createLineString(new Coordinate[] {
      new Coordinate(-1, 0),
      new Coordinate(1, 0), // 2
      new Coordinate(1, 1), // 1
      new Coordinate(4, 5), // 5 - thanks, pythagorus
  });

  @Test
  public void willDoNothingIfGivenASingleEdgedLineStringThatIsLessThanTheDistance() {
    assertEquals(Arrays.asList(singleEdgeLineString), op.apply(singleEdgeLineString, 2.1));
  }

  @Test
  public void willDoNothingIfGivenAMultiEdgedLineStringThatIsLessThanTheDistance() {
    assertEquals(Arrays.asList(doubleEdgeLineString), op.apply(doubleEdgeLineString, 3.1));
    assertEquals(Arrays.asList(manyEdgeLineString), op.apply(manyEdgeLineString, 8.1));
  }

  @Test
  public void willDoNothingIfGivenAMultiEdgedLineStringThatIsEqualToTheDistance() {
    assertEquals(Arrays.asList(doubleEdgeLineString), op.apply(doubleEdgeLineString, 3.0));
    assertEquals(Arrays.asList(manyEdgeLineString), op.apply(manyEdgeLineString, 8.0));
  }

  @Test
  public void willReturnMultipleSegmentsIfASingleEdgedLineStringIsGreaterThanTheDistance() throws Exception {
    assertEquals(
        Arrays.asList(
          gf.createLineString(new Coordinate[] {
              new Coordinate(-1, 0),
              new Coordinate(0, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(0, 0),
              new Coordinate(1, 0)
          })
        ),
        op.apply(singleEdgeLineString, 1));

    assertEquals(
        Arrays.asList(
          gf.createLineString(new Coordinate[] {
              new Coordinate(-1, 0),
              new Coordinate(-.25, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(-.25, 0),
              new Coordinate(0.5, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(0.5, 0),
              new Coordinate(1, 0)
          })
        ),
        op.apply(singleEdgeLineString, 0.75));
  }

  @Test
  public void aDoubleEdgedLineStringWithNicelyDividingDistance() throws Exception {
    assertEquals(
        Arrays.asList(
          gf.createLineString(new Coordinate[] {
              new Coordinate(-1, 0),
              new Coordinate(0, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(0, 0),
              new Coordinate(1, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(1, 0),
              new Coordinate(1, 1)
          })
        ),
        op.apply(doubleEdgeLineString, 1));
  }

  @Test
  public void aDoubleEdgedLineStringWithShortNonDividingDistance() throws Exception {
    assertEquals(
        Arrays.asList(
          gf.createLineString(new Coordinate[] {
              new Coordinate(-1, 0),
              new Coordinate(-.25, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(-.25, 0),
              new Coordinate(0.5, 0)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(0.5, 0),
              new Coordinate(1, 0),
              new Coordinate(1, 0.25)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(1, 0.25),
              new Coordinate(1, 1)
          })
        ),
        op.apply(doubleEdgeLineString, 0.75));
  }

  @Test
  public void aDoubleEdgedLineStringWithLongNonDividingDistance() throws Exception {
    assertEquals(
      Arrays.asList(
        gf.createLineString(new Coordinate[] {
            new Coordinate(-1, 0),
            new Coordinate(1, 0),
            new Coordinate(1, 0.25)
        }),
        gf.createLineString(new Coordinate[] {
            new Coordinate(1, 0.25),
            new Coordinate(1, 1)
        })
      ),
      op.apply(doubleEdgeLineString, 2.25));
  }

  @Test
  public void aManyEdgedLineStringWithLongNonDividingDistance() throws Exception {
    assertEquals(
        Arrays.asList(
          gf.createLineString(new Coordinate[] {
              new Coordinate(-1, 0),
              new Coordinate(1, 0),
              new Coordinate(1, 0.25)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(1, 0.25),
              new Coordinate(1, 1),
              new Coordinate(1.9, 2.2),
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(1.9, 2.2),
              new Coordinate(3.25, 4)
          }),
          gf.createLineString(new Coordinate[] {
              new Coordinate(3.25, 4),
              new Coordinate(4, 5),
          })
        ),
        op.apply(manyEdgeLineString, 2.25));
  }

}
