/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Collections;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.rl.agg.BaseAggregationFunctionTest;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

public class ToListTest extends BaseAggregationFunctionTest {

  @Before
  public void setup() {
    project.add(AggregationFunction.asFunction(new ToListFunction()).builtin("to_list", Category.LANGUAGE));
  }

  @Test
  public void canAggregateValueFromRowsToAList() throws Exception {
    this.assertSerialAndParallelEqual = false;
    Struct inputType = norm(Types.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("to_list(value)")).get();

    Type resultType = RSList.create(Types.INTEGER);
    assertThat(realized.getResultType(), is(resultType));

    // note that we have an identity value - yay, this is now possible that we have an empty list type
    assertThat(this.process(), is(Collections.emptyList()));

    // nothing fancy here, we just want to collect the values
    tuples = tuples(inputType, "[4]", "[1]", "[9]", "[5]");
    process();
    assertThat((List<?>) serialResult, containsInAnyOrder(4L, 1L, 9L, 5L));
    assertThat((List<?>) parallelResult, containsInAnyOrder(4L, 1L, 9L, 5L));
  }

  @Test
  public void canAggregateValueFromNullableRowsToAList() throws Exception {
    this.assertSerialAndParallelEqual = false;
    Struct inputType = norm(Nullable.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("to_list(value)")).get();

    Type resultType = RSList.create(Nullable.INTEGER);
    assertThat(realized.getResultType(), is(resultType));

    // note that we have an identity value - yay, this is now possible that we have an empty list type
    assertThat(this.process(), is(Collections.emptyList()));

    // nothing fancy here, we just want to collect the values
    tuples = tuples(inputType, "[4]", "[null_of('integer')]", "[1]", "[9]", "[5]");
    process();
    assertThat((List<?>) serialResult, containsInAnyOrder(is(4L), nullValue(), is(1L), is(9L), is(5L)));
    assertThat((List<?>) parallelResult, containsInAnyOrder(is(4L), nullValue(), is(1L), is(9L), is(5L)));
  }


}
