/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import static nz.org.riskscape.engine.GeoHelper.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.junit.Test;
import org.junit.Before;
import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.GeometryMatchers;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.gt.LatLongGeometryHelper;
import nz.org.riskscape.engine.gt.NZTMGeometryHelper;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

@SuppressWarnings("unchecked")
public class SegmentTest extends BaseExpressionRealizerTest {

  IdentifiedFunction segment;
  NZTMGeometryHelper nzHelper = new NZTMGeometryHelper(project.getSridSet());
  LatLongGeometryHelper latLongHelper = new LatLongGeometryHelper(project.getSridSet());

  @Before
  public void setup() {
    segment = new Segment().asFunction().builtin("segment", Category.LANGUAGE);
    project.getFunctionSet().add(segment);
  }

  @Test
  public void segmentRetainsCrsIfPresent() {
    Struct childType = Struct.of("geom", Referenced.of(Types.POLYGON, nzHelper.getCrs()));
    Struct inputType = Struct.of("geom", Nullable.of(Referenced.of(Types.POLYGON, latLongHelper.getCrs())),
        "child", Nullable.of(childType));

    Geometry nzBox = nzHelper.box(0, 0, 50, 50);
    Geometry latLong = latLongHelper.reproject(nzBox);
    assertThat((List<? extends Geometry>) evaluate("segment(geom, 100)", Tuple.ofValues(inputType, latLong)),
        contains(
            // the geom is changed just a little because there is reprojection involved in segmenting a
            // lat/long geom
            geometryMatch(latLong, DEGREE_TOLERANCE_NEAREST_CM)
        )
    );
    Type expecteResultType = Nullable.of(RSList.create(Referenced.of(Types.GEOMETRY, latLongHelper.getCrs())));
    assertThat(realized.getResultType(), is(expecteResultType));

    // check that we get some struct -> geom coercing
    assertThat(
        (List<? extends Geometry>)evaluate("segment(child, 100)",
            Tuple.ofValues(inputType, null, Tuple.ofValues(childType, nzBox))
        ),
        contains(
            GeometryMatchers.geometryTopoMatch(nzBox)
        )
    );
    expecteResultType = Nullable.of(RSList.create(Referenced.of(Types.GEOMETRY, nzHelper.getCrs())));
    assertThat(realized.getResultType(), is(expecteResultType));

    // now with no referencing at all
    inputType = Struct.of("geom", Types.POLYGON);
    assertThat((List<? extends Geometry>)evaluate("segment(geom, 100)", Tuple.ofValues(inputType, latLong)),
        contains(
            // the geom is changed just a little because there is reprojection involved in segmenting a
            // lat/long geom
            geometryMatch(latLong, DEGREE_TOLERANCE_NEAREST_CM)
        )
    );
    expecteResultType = RSList.create(Types.GEOMETRY);
    assertThat(realized.getResultType(), is(expecteResultType));
  }

  @Test
  public void handlesBadArguments() {
    Struct inputType = Struct.of("geom", Types.GEOMETRY);

    assertThat(realizeOnly("segment()", Struct.EMPTY_STRUCT), Matchers.failedResult(
        is(ArgsProblems.get().realizableDidNotMatch(segment, Collections.emptyList()))
    ));

    assertThat(realizeOnly("segment('foo', 100)", Struct.EMPTY_STRUCT), Matchers.failedResult(
        is(ArgsProblems.get().realizableDidNotMatch(segment, Arrays.asList(Types.TEXT, Types.INTEGER)))
    ));

    assertThat(realizeOnly("segment(geom, 'foo')", inputType), Matchers.failedResult(
        is(ArgsProblems.get().realizableDidNotMatch(segment, Arrays.asList(Types.GEOMETRY, Types.TEXT)))
    ));

    // extra args
    assertThat(realizeOnly("segment(geom, 100, 'foo')", inputType), Matchers.failedResult(
        is(ArgsProblems.get().realizableDidNotMatch(segment, Arrays.asList(Types.GEOMETRY, Types.INTEGER, Types.TEXT)))
    ));
  }

}
