/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.List;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.RelationType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;

/**
 * Tests the scalar/beta version of to_list(). See {@link ToListTest} for the aggregate version.
 */
public class RelationToListTest extends BaseExpressionRealizerTest {

  Struct struct = Struct.of("bar", Types.INTEGER);
  ListRelation relation = ListRelation.ofTypedValues(struct, 1L, 2L, 3L, 4L, 5L);
  Tuple input = Tuple.ofValues(Struct.of("foo", new RelationType(struct)), relation);
  List<?> listResult;

  @Before
  public void setup() {
    project.getFunctionSet().add(new ToListFunction().builtin("to_list", Category.LANGUAGE));
  }

  @Test
  public void canTurnRelationIntoList() throws Exception {
    evaluate("to_list(foo)", input);
    if (!realizationProblems.isEmpty()) {
      fail(Problem.debugString(realizationProblems));
    }
    assertThat(realized.getResultType(), is(RSList.create(struct)));
    assertThat(evaluated, instanceOf(List.class));
    listResult = (List<?>) evaluated;

    assertThat(listResult.iterator().next(), is(Tuple.ofValues(struct, 1L)));
    assertThat(listResult.get(2), is(Tuple.ofValues(struct, 3L)));
    assertThat(listResult.size(), is(5));
  }

  @Test
  public void canGetSensibleErrorForBadArgs() throws Exception {
    evaluate("to_list()", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ArgsProblems.get().wrongNumber(1, 0))
    )));

    FunctionArgument expectedArg = new ToListFunction().getArguments().get(0);
    evaluate("to_list(baz)", Tuple.ofValues(Struct.of("baz", Types.FLOATING), 1.0D));
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(TypeProblems.get().mismatch(expectedArg, RelationType.WILD, Types.FLOATING)))));
  }
}
